/*$Id: apgen.c,v 1.37 1998/02/19 19:10:09 pbellino Exp $*/
/*
 *****************************************************************************
 **         (C) Copyright 1993, Xyplex, Inc. All Rights Reserved.           **
 *****************************************************************************
 **                           a p g e n . c                                 **
 *****************************************************************************
 **                                                                         **
 **  Description :  This tool reads in a binary parameter file, uncompresses**
 **                 it if necessary, and converts it to its' ASCII          **
 **                 equivalent.                                             **
 **                                                                         **
 **  History     :  Who      What                                           **
 **  06//03/93      BJS/GJG  Original                                       **
 **                                                                         **
 *****************************************************************************
 */

/*
 **  WORRY ABOUT:
 **
 **     *  network -> host order conversions
 **     *  alignment issues"
 */

/*
 **      T A B L E   O F   C O N T E N T S 
 **
 ** Usage() - prints optionStrs ( Table of keyword options for this program )
 **
 ** GetWopt() - 
 **      will search a table of character strings, to see if the given argument
 **      matches the keyword. 
 **
 ** GetWordOpt() - 
 **     uses GetWopt() to get all options, and set appropriate flags 
 **     (displayXxxx) signifying that this option has been chosen.  It also 
 **     processes sub arguments if any are specified.  This utilizes the options
 **     charcter table, and coresponding optionIndices in apgen.h.
 **
 ** ReadParamFileAndUncompress() - 
 **     This routine will read in the parameter file and uncompress it if 
 **     necessary.
 **
 ** main() -
 **     main is responsible for determining the options, getting the 
 **     parameter file uncompressed, and then processing each individual
 **     parameter segment within the parameter file.  
 **     Whenever possible, we process all options chosen with the parameter
 **     segment to be printed with that option. This is done by calling 
 **     the routine to process the data associated with the parameter 
 **     segment, and then the routine to process the given option.
 **     Since not ALL options correspond with a parameter segment, 
 **     in certain cases it is necessary to set up a pointer
 **     to the necessary structure, so it can be processed after the rest
 **     of the parameter file has been processed.
 **     This is done for the ports, and parameter segments.
 **     after all processing is complete, the ProcessKeywordOptions routine
 **     is called.
 **
 ** ProcessKeywordOptions() - 
 **     This routine utilizes numerous routines for each of the keyword options.
 **     In certain cases it is necessary to check to see if the data was 
 **     displayed with a parameter segment, if not, then  we may process it.  
 **     Processing a keyword option consists of making sure that the option has
 **     been chosen, and calling the corresponding routine to process the 
 **     option data that will not been printed by another option.
 **     NOTE: Some data may be printed with another option. ie. if the 
 **     kerberos option was chosen along with either the server or port option.
 **     The server and port option override all other options.  Thus, the
 **     server or port portion of the kerberos data will be printed with 
 **     those options, and not the kerberos option, but we still must print the
 **     other portion.  This does not, however, include the parameter segment
 **     data.  This would have been processed earlier.
 **
 ** ProcessXxxxxOption() -
 **     These are the routines that that will process the given option.
 **     These will determine if the option has been chosen, and call the proper 
 **     routine to print the data.
 **     The actual routines to print the data reside in print_param.c.  They 
 **     are called with a pointer to the beginning of the structure.  This holds
 **     for all except for the routines to print the port data, which will be 
 **     described below.
 **     The routines in print_param.c in general correspond to the options.
 **     As such if an option includes or overrides( as port and server ) other
 **     options then it must call all the routines necessary to print the data
 **
 ** PORT
 **     The port routines are a special case.  They're processing does not 
 **     follow the normal flow, nor does the printing.  
 **     Port information may be disaplayed in one of two ways: in a port list
 **     (ie. 1-4, 7-12), or in the verbose mode where each port is shown
 **     individually.
 **     There are OFFSET_TABLES associated with the each keyword option that 
 **     reside in apgen.h. There is also an INDEX_TABLE that contains all the
 **     OFFSET_TABLES with the routine in print_param.c to call when it 
 **     is necessary to print the data.
 **     
 **     When Necessary to print any or all of the port data, either verbosely or
 **     normal the routine GeneratePortInfo() is used.  This routine takes
 **     as arguments the INDEX_TABLE, an index to the OFFSET_TABLE to be 
 **     printed (INDEX_ALL specifies all), the ports to do it for (portBits, 
 **     which is set up in GetWordOptions()), and wether or not to display it 
 **     verbosely.
 **
 ** EXAMPLE
 **      Let's say we wanted to add a new parameter segment, and an option for 
 **      this.  We'll assume that there is data in the server,and port parameter
 **      segment associated with this also.
 **      1. We need to add the new option the the options character table, 
 **         and optionIndices enumeration.  Then we need to add the global 
 **         displayFoo flag. for the option, and set it in GetWordOptions()
 **         when it has been encountered.
 **         Add to the optionStr what the option is and does.
 **      2. We need to add a routine to print the associated server fields 
 **         in print_param.c.  Or we may add it to a group that already exists.
 **      3. We need to add a routine to print the associated port fields in
 **         print_param.c, or we may add it to an exiting group.
 **      4. If we added the port fields to an existing group, then we need only
 **         add the given offset of the port fields into the appropriate 
 **         OFFSET_TABLE.  Otherwise, we need to create a new offset table,
 **         and add the the offset table, and routine to call to the INDEX_TABLE
 **      4b Modify GenerateCondensedPortInfo and GenerateVerbosePortInfo to
 **         use the new port table if new port data is not contained within
 **         the PTCHAR_STR. The new portTable is initialized in apgen.c
 **      5. If we did not add the port or server fields to an existing group, 
 **         then we need to add a routine to process the option.  This will 
 **         call the print routines for the server, and the port.
 **      6. Now we need to add to main() the processing of the parameter segment
 **         First a routine must be added to print the Parameter segment data,
 **         then this can be called.  If we've added routines to process
 **         the server and port data, then we must call the routine to process
 **         the option.
 **      7. For new port parameter segment, add a new portTable and portNumber
 **         index in apgen.c. Fill in the port table as the new parameter 
 **         segments are encountered. For printing port data, modify 
 **         GenerateCondensedPortInfo and GenerateVerbosePortInfo to
 **         use the new port table instead of default offset table.
 **      8. In ProcessKeywordOptions() if a routine has been added to process 
 **         option, then we need to call it if it hasn't been printed. Otherwise
 **         it will be printed with the port and server data.
 **      
 **      9. Add the new parameter segment to gen_apgen_const and to
 **         gen_apgen_hdr.awk. This creates the new parameter segment definitions
 **         used by the header files and param_data.h. Use gen_apgen_const 
 **         whenever you need a definition from main tree header file instead of
 **         duplicating constant in apgen.h 
 **      
 **     10. After you complete apgen, make sure to test it by running apgen 
 **         against a parameter file, which creates an apgen script, and then
 **         using the SCRIPT command to define the parameters which you created
 **         back on the unit. Also, try testing with a 1 Meg unit. Try with 
 **         all/most features disabled and then with them enabled.
 */

#include <errno.h>
#include <stdio.h>
#include <malloc.h>
#include <time.h>
#include <fcntl.h>
#include "apgen.h"

/*
 ** Software version number
 */

char *softwareVersion = "Version 1.9";

/*
 ** Function Declarations
 */

void       Usage();
void       GetOptions();
void       Log();
void       LogSysError();
void       ProcessKeywordOptions();
void       ProcessSlipOption();
void       ProcessSessionOption();
void       ProcessPPPOption();
void       ProcessLineOption();
void       ProcessTelnetOption();
void       ProcessDaemonOption();
void       ProcessParamServerOption();
void       ProcessSnmpOption();
void       ProcessXRemoteOption();
void       ProcessKerberosOption();
void       ProcessMenuOption();
void       ProcessPortOption();
void       ProcessServerOption();
void       ProcessSecurIdOption();
void       ProcessArapOption();
void       ProcessIpxOption();
void       ProcessIpxRouteOption();
void       ProcessIpxFilterOption();
void       ProcessSecurityOption();
void       ProcessIpOption();
void       ProcessIpFilterOption();
void       ProcessFeaturesOption();
void       ProcessModemRelatedOption();
void       ProcessRadiusOption();
extern void       PrintServerXRemoteData();
extern void       PrintServerFeaturesData();
extern void       PrintPortSlipData();
extern void       PrintPortPPPData();
extern void       PrintPortLineData();
extern void       PrintPortSessionData();
extern void       PrintPortTelnetData();
extern void       PrintPortXRemoteData();
extern void       PrintPortKerberosData();
extern void       PrintPortOldMenuData();
extern void       PrintPortNestedMenuData();
extern void       PrintPortSecurityData();
extern void       PrintPortArapData();
extern void       PrintServerArapData();

/*
 ** Global Storage
 */

char     *appName;                 /*  Name of this program                  */
char     *inputFileName = "\0";    /*  Parameter file                        */
char     *outputFileName = "\0";   /*  File to place results (optional)      */
char     *validationString = "Load File";  /* Parameter file validation */
int       paramDataSize;           /*  Size of parameter data                */
char      portString[200];        
char      portBits[TBITS_PORT_LIST_SIZE];
char     *portStringPtr = portString;
int       portNumber = 0;          /* index/count of port segments */
int       arapPortNumber = 0;       /* index/count of arap port segments */
int       controlledPortNumber = 0; /* index/count of ctrl port segments */
int       pppPortNumber = 0;       /* index/count of ppp port segments */
int       cclPortNumber = 0;       /* index/count of ccl port segments */
int       expandedPortNumber = 0;  /* index/count of expanded port segments */
char     *portTable[TBITS_PORT_LIST_SIZE * 8];     /* table of port segments */
char     *arapPortTable[TBITS_PORT_LIST_SIZE * 8]; /* arap port segments */
char     *controlledPortTable[TBITS_PORT_LIST_SIZE * 8]; /* controlled port segments */
char     *pppPortTable[TBITS_PORT_LIST_SIZE * 8]; /* ppp port segments */
char     *cclPortTable[TBITS_PORT_LIST_SIZE * 8]; /* ccl port segments */
char     *expandedPortTable[TBITS_PORT_LIST_SIZE * 8]; /* expanded port segments */
char     *serverPtr;
/* PJB Added new pointer in support of the 8 extra SNMP gets, sets and traps- V6-0-3 release. 
 * Fixed bug for pre V6-0-4 - V6-0-3 paramter files.
 */
char     *snmpExpServerPtr = 0;
char     *arapServerPtr;

int       ipxInterfaceNumber = 0; /* index/count of ipx port segments */
char     *ipxInterface[TBITS_PORT_LIST_SIZE * 8]; /* ipx port segments */
int       ipxRouteNumber = 0; /* index/count of ipx static route segments */
char     *ipxRoute[IPX_MAX_STATIC_ROUTES]; /* ipx static route segments */
int       ipxServiceNumber = 0; /* index/count of ipx static service segments */
char     *ipxService[IPX_MAX_STATIC_SERVICES]; /* ipx static service segments */
int       ipxRipFilterNumber = 0; /* index/count of ipx rip filter segments */
char     *ipxRipFilter[IPX_MAX_RIP_FILTERS]; /* ipx rip filter segments */
int       ipxSapFilterNumber = 0; /* index/count of ipx sap filter segments */
char     *ipxSapFilter[IPX_MAX_SAP_FILTERS]; /* ipx sap filter segments */
int       ipxAddressFilterNumber = 0; /* index/count of ipx Address filter segments */
char     *ipxAddressFilter[IPX_MAX_SRC_DEST_FILTERS]; /* ipx address filter segments */
int       ipFilterNumber = 0; /* index/count of ip filter segments */
char     *ipFilter[IP_MAX_FILTERS]; /* ip filter segments */

char      portXmitBuff[TBITS_PORT_LIST_SIZE];
char      portXmitIdle[TBITS_PORT_LIST_SIZE];
unsigned int initPortArrays = TRUE;

unsigned int   mop = FALSE;               /* Signifies parameter file is mop */
unsigned char  SoftwareVersionMajor;
unsigned char  SoftwareVersionMinor;
unsigned char  SoftwareVersionEco;
unsigned char  SoftwareVersionType;
unsigned char  SoftwareVersionPrerelease;
unsigned char  SoftwareProduct;
unsigned int   HardwareType;
unsigned int   kerberosParamsSeen = FALSE;
unsigned int   securIdParamsSeen = FALSE;
unsigned int   menuParamsSeen = FALSE;
unsigned int   paramServerParamsSeen = FALSE;
unsigned int   securityParamsSeen = FALSE;
unsigned int   xremoteParamsSeen = FALSE;
unsigned int   arapParamsSeen = FALSE;
unsigned int   ipxInterfaceParamsSeen = FALSE;
unsigned int   ipxFilterParamsSeen = FALSE;
unsigned int   ipxRouteParamsSeen = FALSE;
unsigned int   ipFilterParamsSeen = FALSE;
unsigned int   pppParamsSeen = FALSE;
unsigned int   cclParamsSeen = FALSE;
unsigned int   expandedPortParamsSeen = FALSE;
int wind = 0; /* keyword  argv index */
char *woptarg;   /* pointer to  */
int werr = FALSE;

/*
 ** Flags which control how much information is displayed.
 */

unsigned int displayAll           = FALSE;
unsigned int displayArap          = FALSE;
unsigned int displayDaemons       = FALSE;
unsigned int displayDomain        = FALSE;
unsigned int displayVerbose       = FALSE;
unsigned int displayKerberos      = FALSE;
unsigned int displayLine          = FALSE;
unsigned int displayLpd           = FALSE;
unsigned int displayLoad          = FALSE;
unsigned int displayManager       = FALSE;
unsigned int displayMenu          = FALSE;
unsigned int displayNVS           = FALSE;
unsigned int displayParamServer   = FALSE;
unsigned int displayPort          = FALSE;
unsigned int displayPPP           = FALSE;
unsigned int displayRadius        = FALSE;
unsigned int displayRotary        = FALSE;
unsigned int displayRouteIp       = FALSE;
unsigned int displayScript        = FALSE;
unsigned int displaySecurId      = FALSE;
unsigned int displaySecurity      = FALSE;
unsigned int displayServer        = FALSE;
unsigned int displayService       = FALSE;
unsigned int displaySession       = FALSE;
unsigned int displaySlip          = FALSE;
unsigned int displaySnmp          = FALSE;
unsigned int displayTelnet        = FALSE;
unsigned int displayTN3270        = FALSE;
unsigned int displayXRemote       = FALSE;
unsigned int displayXPrinter      = FALSE;
unsigned int displayFeatures      = FALSE;
unsigned int displayIp            = FALSE;
unsigned int displayIpx           = FALSE;
unsigned int displayIpxFilters    = FALSE;
unsigned int displayIpxStaticData = FALSE;
unsigned int displayIpFilters     = FALSE;

/*
 ** Flags that control how much information is displayed based on data in the 
 ** server parameter segment.
 */

unsigned int securityEnabled      = FALSE;
unsigned int arapEnabled          = FALSE;
unsigned int accountingEnabled    = FALSE;
unsigned int tn3270Enabled        = FALSE;
unsigned int snmpEnabled          = FALSE;
unsigned int pppEnabled           = FALSE;
unsigned int nestedMenuEnabled    = FALSE;
unsigned int telnetEnabled        = FALSE;
unsigned int latEnabled           = FALSE;
unsigned int kerberosEnabled      = FALSE;
unsigned int kerberos5Enabled      = FALSE;
unsigned int multiSessionsEnabled = FALSE;
unsigned int unixEnabled          = FALSE;
unsigned int lpdEnabled           = FALSE;
unsigned int menuEnabled          = FALSE;
unsigned int scriptEnabled        = FALSE;
unsigned int securIdEnabled       = FALSE;
unsigned int xremoteEnabled       = FALSE;
unsigned int xprinterEnabled      = FALSE;
unsigned int ipxEnabled           = FALSE;
unsigned int ipxFilteringEnabled  = FALSE;
unsigned int ipFilteringEnabled   = FALSE;
unsigned int controlledPortsEnabled = FALSE;
unsigned int apdEnabled           = FALSE;
unsigned int radiusEnabled        = FALSE;
char   *loadTable[MAX_NETWORK_DEVICES];

/*
 ** External variables.
 */

extern int       errno;
extern int       optind;
extern char     *optarg;
extern char     *options[];


/*
 ** External functions.
 */
extern void      PrintParamHeader();
extern void      PrintLpdParams();
extern void      PrintServerParams();
extern void      PrintDomainParams();
     
/*
 ** Usage and Option strings.
 */
    
char *usageStr  = "\nUsage: %s [options] parameter-file [optional-output-file]\n";
     
char *optionStrs[] = {

"options are:\n",
"    -all               display all information\n",
"    -arap              display arap information\n",
"    -daemons           display daemons data\n",
"    -domain            display domain data\n",
"    -filter[:ip | ipx] display ipx filter data\n",
"    -features          display server features data\n",
"    -ip                display internet data\n",
"    -ipx               display ipx data\n",
"    -kerberos          display kerberos data\n",
"    -lineedit          display port line data\n",
"    -lpd               display LPD data\n",
"    -manager           display manager data\n",
"    -menu              display old menu and nested menu data\n",
"    -nvs               display NVS data\n",
"    -parameter         display parameter server data\n",
"    -port[:<port>]     display port data (-port:all or -port display all ports)\n",
"    -ppp               display ppp data\n",
"    -radius            display radius data\n",
"    -rotary            display rotary data\n",
"    -route[:ip | ipx]  display route data (ipx also includes service data)\n",
"    -script            display script data\n",
"    -securid           display Secure ID information\n",
"    -security          display internet security data\n",
"    -server            display server data\n",
"    -service           display service data\n",
"    -session           display port session data\n",
"    -slip              display port slip data\n",
"    -snmp              display internet snmp data\n",
"    -telnet            display port telnet data\n",
"    -tn3270            display tn3270 data\n",
"    -verbose           display port information individually, not in list\n",
"    -xprinter          display xprinter data\n",
"    -xremote           display xremote data\n",
};

/*
 ** Usage()
 **
 ** Prints out the an informational message and the usage string
 ** for this process, then calls exit.
 **
 ** Arguments:
 **
 **     message         The informational message to be printed before
 **
 ** Exits:
 **
 **     Every time.
 */

void Usage(message)
     char    *message;
{
    int numOptStrs, j;
    
    if (message)
        {
        fprintf(stderr, "\n%s \n", message);
        }
    
    fprintf(stderr, "\nAPGEN %s\n", softwareVersion);
    fprintf(stderr, usageStr, appName);
    
    numOptStrs = sizeof(optionStrs) / sizeof(optionStrs[0]);
    
    for (j = 0; j < numOptStrs; j++)
        {
        fprintf(stderr, optionStrs[j]);
        }
    
    exit(0);
    }

/*
 ** Log(message)
 **
 ** Prints out the current timestamp,program name, and message.
 **
 ** Arguments:
 **
 **     message         The message to be printed.
 */

void Log(message)
     char    *message;
{
    time_t      t;
    char        *timeStr;
    int         i;
    

    t = time(0);
    timeStr = ctime( &t );

    for (i=0; i<strlen(timeStr); i++)
	{
	if (timeStr[i] == '\n')
	    {
	    timeStr[i] = ':';
	    break;
	    }
	}
    
    fprintf(stderr, "%s %s\n", timeStr, message);


    }

/*
 ** LogSysError(message)
 **
 ** Prints out the error message and message related to the
 ** value in the global variable "errno".
 **
 ** Arguments:
 **
 **     message         The error message to be printed.
 */

void LogSysError(message)
     char    *message;
{
    perror(message);
    }

/*
 ** position()
 */

int position( s, c, max)
     char *s;
     char  c;
     int max;
{
    int  loop =0;
    char *temp = s;
    
    while(( *temp != c) && (temp != '\0') && (loop <max))
	{
	temp++;
	loop++;
	}
    if (*temp == c)
	{
	return ((int) ((unsigned int)temp - (unsigned int)s));
	}
    else
	{
	return -1;
	}
    }


int GetWOpt( argc, argv, optTable)
     int    argc;
     char  *argv[];
     char  *optTable[];
{
    int loop = 1, loop2 = 0;
    int  argsEqual = 0;
    int  argsFound = 0;
    int  optionHasArguments = FALSE;
    int  keywordHasArguments = FALSE;
    int  optionIndex = 0;
    
    wind++;
    if (wind == argc)
	{
	/* 
	 ** We're out of options 
	 */
	return -1;
	}
    
    if ((argv[wind][0] == '-') && (wind < argc))
	{
	/*
	 * Now we have the argument, lets see which one it is
	 */
	optionIndex = 0;

	/* 
	 ** Loop through all keywords in the table
	 */

	while ((strcmp((char *)optTable[optionIndex], END_OPTION) != 0) && 
	      !argsFound)
	    {
	    /*
	     ** arguments will follow immediately after the keyword, only
	     ** separated by a :
	     */
	    argsEqual = 0;
	    argsFound = 0;	    
	    if ((keywordHasArguments = position(optTable[optionIndex], ':',
					 (strlen(optTable[optionIndex]) +1)))
		!= -1)
		{
		/* compare the argument with this keyword */
		if ((argsEqual = strncmp(optTable[optionIndex], 
				    (char *)&argv[wind][1],
				       keywordHasArguments)) == 0)
		    {
		    argsFound = TRUE;
		    /*
		     ** option should have arguments, does it?
		     */
		    
		    if ((optionHasArguments = position(argv[wind], ':',
						(strlen(argv[wind]) +1))) 
			!= -1)
			{
			woptarg = (char *) &argv[wind][optionHasArguments+1];
			}
		    else
			{
			/*
			 ** check to make sure there is nothing after the opt
			 */
			if (argv[wind][keywordHasArguments+1] != '\0')
			    {
			    return (KEYWORD_ERROR);
			    }
			else
			    {
			    woptarg = NULL_VALUE;
			    }
			}
		    }
		}
	    else
		{
		if ((argsEqual = strcmp(optTable[optionIndex], 
				       (char *)&argv[wind][1])) == 0)
		    {
		    argsFound = TRUE;
		    }
		
		}
	    optionIndex++;
	    } /* end while ! argfound */
	if (argsFound)
	    {
	    return (optionIndex -1);
	    }
	else
	    {
	    werr = KEYWORD_ERROR;
	    return (KEYWORD_ERROR);
	    }
	} /* end if there is a '-' */
    else
	{
	if (argv[wind][0] == '?')
	    {
	    return (KEYWORD_USAGE);
	    }
	else
	    {
	    return -1;
	    }
	}
}	    



/*
 ** GetWordOptions(argc, argv)
 */

void GetWordOptions(argc, argv)
     int    argc;
     char *argv[];
{
    int                  keyword;
    int                  numArgs, outputFd;
    char                *appName, *portArg;
    struct hostent      *hostEnt;
    char                optionArg[ 80];

    /*
     ** need to specify that we're showing all ports, unless user overrides
     */

    bits_set(portBits, PORT_ZERO,( (TBITS_PORT_LIST_SIZE * 8) - 1));


    /*
     ** All options must begin with a '-'
     */
    while ((keyword = GetWOpt(argc, argv, options )) != -1)
	{
	switch (keyword)
	    {
	case ALL_INDEX:
	    {
	    displayAll = TRUE;
	    break;
	    }
	case ARAP_INDEX:
	    {
	    displayArap = TRUE;
	    break;
	    }
	case DAEMONS_INDEX:
	    {
	    displayDaemons = TRUE;
	    break;
	    }
	case DOMAIN_INDEX:
	    {
	    displayDomain = TRUE;
	    break;
	    }
	case FEATURES_INDEX:
	    {
	    displayFeatures = TRUE;
	    break;
	    }
	case INTERNET_INDEX:
	    {
	    displayIp = TRUE;
	    break;
	    }
	case KERBEROS_INDEX:
	    {
	    displayKerberos = TRUE;
	    break;
	    }
	case LINE_EDIT_INDEX:
	    {
	    displayLine = TRUE;
	    break;
	    }
	case LPD_INDEX:
	    {
	    displayLpd = TRUE;
	    break;
	    }
	case MENU_INDEX:
	    {
	    displayMenu = TRUE;
	    break;
	    }
	case MANAGER_INDEX:
	    {
	    displayLoad = TRUE;
	    displayManager = TRUE;
	    break;
	    }
	case NVS_INDEX:
	    {
	    displayNVS = TRUE;
	    break;
	    }
	case PARAMETER_INDEX:
	    {
	    displayParamServer = TRUE;
	    break;
	    }
	case PORT_INDEX:
	    {
	    /*
	     * Anything immediately folllowing the port number will be 
	     * interpreted as part of the port number, flag this as an error.
	     */
	    
	    if (!displayPort)
		{
		/*
		 ** we must clear all the port bits, that we set 
		 ** at the beginning of this routine.
		 ** This way will only print ports user tells us to
		 */
		bits_clear(portBits, PORT_ZERO,
			   ( (TBITS_PORT_LIST_SIZE * 8) - 1));
		}

	    displayPort = TRUE;
	    
	    if (woptarg != NULL_VALUE)
		{
		portArg = woptarg;
		if (*portArg == '\0')
		    {
		    Usage("Illegal option.");
		    }
		}
	    else 
		{
		/*
		 ** This means there was no : or argument specified.
		 ** defualt is to process all bits in this case
		 ** break out after doing this.
		 */
		bits_set(portBits, 0, ((TBITS_PORT_LIST_SIZE *8)-1));
		break;   
		}
	    
	    if (strcmp("all", portArg) == 0)
		{
		bits_set(portBits, 0, ((TBITS_PORT_LIST_SIZE *8)-1));
		break;
		}
	    
	    portStringPtr = portString;		    
	    while((*portArg >= '0') && (*portArg <='9') &&(*portArg != '\0'))
		{
		sprintf(portStringPtr,"%c",*portArg);
		portStringPtr = portString + strlen(portString);
		portArg++;
		}
	    
	    bits_set(portBits,atoi((char *) portString),
		     atoi((char *) portString));
	    if(*portArg != '\0')    /* something is following the number */
		{
		Usage("Illegal option.");
		}
	    break;
	    }
	case PPP_INDEX:
	    {
	    displayPPP = TRUE;
	    break;
	    }
	case ROUTE_INDEX:
	    {
	    if (woptarg != NULL_VALUE)
		{
		portArg = woptarg;
		if (*portArg == '\0')
		    {
		    Usage("Illegal option.");
		    }
		}
	    else 
		{
		/*
		 ** This means there was no : or argument specified.
		 ** defualt is to process all bits in this case
		 ** break out after doing this.
		 */
		displayRouteIp = TRUE;
		break;   
		}
	    
	    if (strcmp("ip", portArg) == 0)
		{
		displayRouteIp = TRUE;
		break;
		}
	    if (strcmp("ipx", portArg) == 0)
		{
		displayIpxStaticData = TRUE;
		break;
		}
	    break;
	    }
	case RADIUS_INDEX:
	    {
	    displayRadius = TRUE;
	    break;
	    }
	case ROTARY_INDEX:
	    {
	    displayRotary = TRUE;
	    break;
	    }
	case SECURID_INDEX:
	    {
	    displaySecurId = TRUE;
	    break;
	    }
	case SECURITY_INDEX:
	    {
	    displaySecurity = TRUE;
	    break;
	    }
	case SERVICE_INDEX:
	    {
	    displayService = TRUE;
	    break;
	    }
	case SERVER_INDEX:
	    {
	    displayServer = TRUE;
	    break;
	    }
	case SESSION_INDEX:
	    {
	    displaySession = TRUE;
	    break;
	    }
	case SLIP_INDEX:
	    {
	    displaySlip = TRUE;
	    break;
	    }
	case SNMP_INDEX:
	    {
	    displaySnmp = TRUE;
	    break;
	    }
	case TELNET_INDEX:
	    {
	    displayTelnet = TRUE;
	    break;
	    }
	case TN3270_INDEX:
	    {
	    displayTN3270 = TRUE;
	    break;
	    }
	case VERBOSE_INDEX:
	    {
	    displayVerbose = TRUE;
	    break;
	    }
	case XREMOTE_INDEX:
	    {
	    displayXRemote = TRUE;
	    break;
	    }
	case XPRINTER_INDEX:
	    {
	    displayXPrinter = TRUE;
	    break;
	    }
	case SCRIPT_INDEX:
	    {
	    displayScript = TRUE;
	    break;
	    }
	case IPX_INDEX:
	    {
	    displayIpx = TRUE;
	    break;
	    }
	case FILTER_INDEX:
	    {
	    if (woptarg != NULL_VALUE)
		{
		portArg = woptarg;
		if (*portArg == '\0')
		    {
		    Usage("Illegal option.");
		    }
		}
	    else 
		{
		/*
		 ** This means there was no : or argument specified.
		 ** defualt is to process all bits in this case
		 ** break out after doing this.
		 */
		displayIpFilters = TRUE;
		break;   
		}
	    
	    if (strcmp("ip", portArg) == 0)
		{
		displayIpFilters = TRUE;
		break;
		}
	    if (strcmp("ipx", portArg) == 0)
		{
		displayIpxFilters = TRUE;
		break;
		}
	    break;
	    }
	case USAGE_INDEX:
	case KEYWORD_USAGE:
	    {
	    Usage((char *)0);
	    }
	case KEYWORD_ERROR:
	    {
	    Usage("Invalid argument(s)\n");
	    }	    
	    }
	}

    numArgs = argc - wind;
    if (numArgs < 1)
	{
	Usage( "Parameter file name missing");
	}
    
    if (numArgs > 2)
	{
	Usage( "Too many arguments");
	}

    /*
     ** The input file is the next argument.
     */

    inputFileName  = argv[wind];

    /*
     ** If there is another argument, its the output file.  
     */

    if (numArgs == 2)
	{
	outputFileName = argv[wind+1];

	if ( (outputFd = open( outputFileName, O_WRONLY | O_CREAT | O_TRUNC )) < 0)
	    {
	    LogSysError( "Unable to open Output file" );
	    Log( "Using stdout" );
	    }

	if ( (fchmod( outputFd, 0766 )) < 0)
	    {
	    LogSysError( "Unable to set file permissions on output file" );
	    }
	
	if ( (dup2( outputFd, 1 )) < 0)
	    {
	    LogSysError( "Unable to dup2() into stdout" );
	    exit( 0 );
	    }
	close( outputFd );
	}
}

/*
 ** ReadParamFileAndUncompress()
 **
 ** Arguments:
 **
 **     fileName        Name of file to read and uncompress.
 **
 ** Returns:
 **
 **     Pointer (char) to buffer containing uncompressed file.
 **
 ** Exits:
 **
 **     When problems are encountered during the read of the parameter file.
 */
char *ReadParamFileAndUncompress( fileName )
     char   *fileName;
{

    FILE           *fd;
    char            paramHeader[PARAM_LOAD_STR_sizelen+ MOP_HEADER_SIZE], *paramFile;
    char            tmpFileName[80], *paramData, *scanFileName,*noPathFileName;
    int             i, index, mopOffset = 0;
    struct stat     fileStat;
    char           *paramHeaderPtr;
    
    /*
     ** Open the file for reading.
     */

    if ( (fd = fopen( fileName, "r" ) ) == (FILE *)0)
	{
	LogSysError( "Error Opening Input File" );
	exit( 0 );
	}

    /*
     ** Read in the header of the file and close the input file.
     */

    for (i=0; i < (PARAM_LOAD_STR_sizelen + MOP_HEADER_SIZE); i++)
	{
	paramHeader[i] = fgetc( fd );
	}
    
    fclose( fd );

    paramHeaderPtr = paramHeader;

    /*
     ** lets check for a mop header.  if the 11th, 12th, 13th, and 14th
     ** bytes are FF FF, and the 19th and 20th are 0, and the 21th and 22th are
     ** 0, and the 20th and 21st are 01 01 then we're MOP.
     */

    if (((unsigned char)paramHeaderPtr[10] == (unsigned char)0xFF) &&
	((unsigned char)paramHeaderPtr[11] == (unsigned char)0xFF) &&
	((unsigned char)paramHeaderPtr[12] == (unsigned char)0xFF) &&
	((unsigned char)paramHeaderPtr[13] == (unsigned char)0xFF) &&
	((unsigned char)paramHeaderPtr[18] == (unsigned char)0x00) &&
	((unsigned char)paramHeaderPtr[19] == (unsigned char)0x00) &&
	((unsigned char)paramHeaderPtr[20] == (unsigned char)0x01) &&
	((unsigned char)paramHeaderPtr[21] == (unsigned char)0x01))
	{
	mop = TRUE;
	mopOffset = MOP_HEADER_SIZE;
	}	

    /*
     ** First thing first, lets check to make sure this is actually a 
     ** parameter file.
     */

    if (strncmp((char *) &paramHeaderPtr[PARAM_LOAD_STR_bs2 + BS2_STR_TAG_validation + mopOffset + 7],
		(char *) validationString, 9) != 0)
	{
	LogSysError("This is not a valid parameter file");
	exit(0);
	}

    /*
     ** Let's get the heck out of here if this parameter file doesn't support 
     ** the new param_load_str.
     */
    
    if (paramHeaderPtr[PARAM_LOAD_STR_differential_hours+ mopOffset] == 0)
	{
	/*
	 ** Let's see if it's a MOP header file, we can check certain
	 ** to see.  
	 */

	LogSysError("Parameter file version not supported by APGEN");
	exit(0);
	}
    
    /*
     ** Uncompress the file, if we need to.
     */
    
    if (paramHeaderPtr[PARAM_LOAD_STR_differential_minutes + mopOffset])
	{
	/*
	 ** let's get the filename to append to the /tmp/uncompressed_
	 ** This is necessary if we have a pathname.
	 ** We'll search for a '/' in the filename, and then set the 
	 ** noPathFileName to the next character.
	 */
	scanFileName = fileName;
	noPathFileName = fileName;
	for (index = 0; index < (strlen(fileName)); index++)
	    {
	    if (*scanFileName == '/')
		{
		noPathFileName = scanFileName + 1;
		}
	    scanFileName++;
	    }
	
	/*
	 ** Create a temp file for the uncompress code.
	 */
	sprintf( tmpFileName, "/tmp/uncompressed_%s", noPathFileName );
	unlink( tmpFileName );
	uncompress( fileName, tmpFileName );
	paramFile = (char *)tmpFileName;
	}
    else
	{
	paramFile = fileName;
	}

    /*
     ** paramFile now contains the name of our uncompressed parameter
     ** file.  Let's read it into a buffer.
     */

    if ( (stat( paramFile, &fileStat )) < 0)
	{
	LogSysError( "Could not 'stat' uncompressed parameter file" );
	exit( 0 );
	}


    if ( (fd = fopen( paramFile, "r" ) ) == (FILE *)0)
	{
	LogSysError( "Error Opening Uncompressed Parameter File" );
	exit( 0 );
	}

    if ( (paramData = (char *)malloc(fileStat.st_size+1)) == 0)
	{
	LogSysError( "Could not malloc() buffer for parameters" );
	exit( 0 );
	}
    
    paramDataSize = fileStat.st_size+1;
    
    for (i=0; i<fileStat.st_size; i++)
	{
	paramData[i] = fgetc( fd );
	}

    fclose( fd );
    
    return( paramData );
    
    }

/*
 ** main()
 **
 ** Main procedure, basically just calls everybody else.
 */
main( argc, argv )
     int    argc;
     char   *argv[];
{
    short  int    paramSegmentType, paramSegmentLength, loadTableIndex = 0;
    char   *paramDataPtr,*startParamDataPtr, errorString[80];
    char   *paramSegmentPtr;
    
    /*
     ** Parse command line options.
     */
    
    appName = argv[0];
    
    GetWordOptions( argc, argv);
    
    /*
     ** Read and uncompress the file (if we need to)
     */

    startParamDataPtr = ReadParamFileAndUncompress( inputFileName );

    /*
     **  Now let's print out all of the data.
     */

    paramDataPtr = startParamDataPtr;
    PrintParamHeader( paramDataPtr );
    paramDataPtr += PARAM_LOAD_STR_sizelen;

    paramSegmentPtr = paramDataPtr;

    while (paramDataPtr < (startParamDataPtr + paramDataSize))
	{
	paramDataPtr += PARAM_MSG_SEGMENT_STR_sizelen;

	paramSegmentType = GetWord( 
	      (char *)&paramSegmentPtr[PARAM_MSG_SEGMENT_STR_type]);

	paramSegmentLength = GetWord( 
     	      (char *)&paramSegmentPtr[PARAM_MSG_SEGMENT_STR_length]);

	switch( paramSegmentType )
	    {
	case PARAM_SEGMENT_END_OF_LIST:             
	    break;

	case PARAM_SEGMENT_SERVER_DATA:             
	    {
	    serverPtr = paramDataPtr;
	    SetServerEnableFlags( paramDataPtr );
	    break;
	    }
/* PJB Added new case statement in support of the 8 extra SNMP gets, sets and traps- V6-0-3 release. */
	case PARAM_SEGMENT_SERVER_SNMP_EXP_DATA:             
	    {
	    snmpExpServerPtr = paramDataPtr;
	/*     SetServerEnableFlags( paramDataPtr ); */
	    break;
	    }

    case PARAM_SEGMENT_SERVER_EXPANDED_DATA:
    	{
     	if (displayAll | displayServer)
		{
		PrintServerExpandedParams( paramDataPtr );
	    }
		break;
	    }
	case PARAM_SEGMENT_SERVICE_DATA:            
	    {
	    if (displayAll | displayService )
		{
		PrintServiceParams( paramDataPtr );
		}
	    break;
	    }
	    
	case PARAM_SEGMENT_PORT_DATA:               
	    {
            /*
             * For MX1608, stop after 8 ports. Dont use dummy ports 9-16. 

             */		
            if ((HardwareType != ROM_CRATE_HW_MX1608) || (portNumber <= portNumberMX1608)) /* SC65492 FBS 092895 */
            {		
	        portTable[portNumber] = paramDataPtr;
	    /*
	     * This processing need to be completed after the portTable is 
	     * set up. so it is done after the entire parameter file is 
	     * processed.
	     */

	        portNumber++;
            }
	    break;
	    }
	    
	case PARAM_SEGMENT_DOMAIN_DATA:             
	    {
	    if (displayAll | displayDomain)
		{
		PrintDomainParams( paramDataPtr );
		}
	    break;
	    }
	    
	case PARAM_SEGMENT_ROUTE_DATA:              
	    {
	    if (displayAll | displayRouteIp)
		{
		PrintRouteParams( paramDataPtr );
		}
	    break;
	    }

	case PARAM_SEGMENT_LOAD_DATA:               
	    {
	    /*
	     * We need to put the address into the load table.
	     * This is because the load data really gets printed with the
	     * manager data.  There is a flag in the manager data which tells
	     * us if we need to comment the load data out...Only problem is 
	     * the load data get's processed first, hence the creation of the
	     * load table.  This is so the PrintManagerParams can print the
	     * load data.
	     */

	    loadTable[ loadTableIndex++ ] =  paramDataPtr;
	    break;
	    }

	case PARAM_SEGMENT_MANAGER_DATA:   
	    {
	    if (displayAll | displayManager)
		{
		PrintManagerParams( paramDataPtr );
		}
	    break;
	    }

	case PARAM_SEGMENT_MENU_DATA:           
	    {
	    if (menuParamsSeen == FALSE)
		{
		ProcessMenuOption();
		}
	    menuParamsSeen = TRUE;
	    if (displayAll | displayMenu)
		{
		PrintMenuDataParams( paramDataPtr );
		}
	    break;
	    }
 
	case PARAM_SEGMENT_MENU_TEXT_DATA:           
	    {
	    if (displayAll | displayMenu)
		{
		PrintMenuTextParams( paramDataPtr );
		}
	    break;
	    }
	    
	case PARAM_SEGMENT_PARAM_SERVER_DATA:       
	    {
	    if (paramServerParamsSeen == FALSE)
		{
		ProcessParamServerOption();
		}
	    paramServerParamsSeen = TRUE;
	    if (displayAll | displayParamServer)
		{
		PrintParamServerParams( paramDataPtr ); 
		}
	    break;
	    }
	    
	case PARAM_SEGMENT_DEFAULT_SECURITY_DATA:              
	    {
	    securityParamsSeen = TRUE;
	    if (displayAll | displaySecurity)
		{
		PrintDefSecurityParams( paramDataPtr ); 
		}
	    ProcessSecurityOption();
	    break;
	    }


	case PARAM_SEGMENT_SECURITY_DATA:              
	    {
	    if (displayAll | displaySecurity)
		{
		PrintSecurityParams( paramDataPtr );
		}
	    break;
	    }

	case PARAM_SEGMENT_KERBEROS_DATA:           
	    {
	    kerberosParamsSeen = TRUE;
	    if (displayAll | displayKerberos)
		{
		PrintKerberosParams( paramDataPtr );
		}
	    ProcessKerberosOption();
	    break;
	    }

        case PARAM_SEGMENT_KERB739_DATA:
	    {
	    if (displayAll | displayKerberos)
		{
		PrintKerberosExtraParams( paramDataPtr );
		}
	    break;	
            }		
        case PARAM_SEGMENT_RADIUS_DATA:
	    {
	    if (displayAll | displayRadius)
		{
		PrintRadiusServerParams( paramDataPtr );
		}
	    break;	
            }		

	case PARAM_SEGMENT_ROTARY_DATA:             
	    {
	    if (displayAll | displayRotary)
		{
		PrintRotaryParams( paramDataPtr );
		}
	    break;
	    }

	case PARAM_SEGMENT_SCRIPT_DATA:             
	    {
	    if (displayAll | displayScript)
		{
		PrintScriptParams( paramDataPtr );
		}
	    break;
	    }

	case PARAM_SEGMENT_XREMOTE_DATA:            
	    {
	    xremoteParamsSeen = TRUE;
	    if (displayAll | displayXRemote)
		{
		PrintXRemoteParams( paramDataPtr );
		}
	    ProcessXRemoteOption();
	    break;
	    }

	case PARAM_SEGMENT_XPRINTER_DATA:           
	    {
	    if (displayAll | displayXPrinter)
		{
		PrintXPrinterParams( paramDataPtr );
		}
	    break;
	    }

	case PARAM_SEGMENT_NVS_DATA:
	    {
	    if (displayAll | displayNVS)
		{
		PrintNVSParams( paramDataPtr );
		}
	    break;
	    }
	    
	case PARAM_SEGMENT_LPD_DATA:                
	    {
	    if (displayAll | displayLpd)
		{
		PrintLpdParams( paramDataPtr );
		}
	    break;
	    }
	    
	case PARAM_SEGMENT_TN3270_DEVICE_DATA:       
	    {
	    if (displayAll | displayTN3270)
		{
		PrintTN3270DeviceParams( paramDataPtr );
		}
	    
	    break;
	    }

	case PARAM_SEGMENT_TN3270_XLAT_DATA:       
	    {
	    if (displayAll | displayTN3270)
		{
		PrintTN3270XlatParams( paramDataPtr );
		}
	    
	    break;
	    }

	case PARAM_SEGMENT_ARAP_SERVER_DATA:
	    {
	    arapServerPtr = paramDataPtr;
	    break;
	    }
	    
	case PARAM_SEGMENT_ARAP_PORT_DATA:
	    {
            if ((HardwareType != ROM_CRATE_HW_MX1608) || (arapPortNumber <= portNumberMX1608)) /* SC65492 FBS 092895 */
            {		
	    arapParamsSeen = TRUE;
	    arapPortTable[arapPortNumber] = paramDataPtr;
	    arapPortNumber++;
            }		
	    break;
	    }
	case PARAM_SEGMENT_SECURID_DATA:
	    {
	    securIdParamsSeen = TRUE;
	    if (displayAll | displaySecurId)
		{
		PrintSecurIdParams( paramDataPtr);
		}
	    ProcessSecurIdOption();
	    break;
	    }
	case PARAM_SEGMENT_CONTROLLED_PORT_DATA:
	    {
            if ((HardwareType != ROM_CRATE_HW_MX1608) || (controlledPortNumber <= portNumberMX1608)) /* SC65492 FBS 092895 */
            {		
	    controlledPortTable[controlledPortNumber] = paramDataPtr;
	    controlledPortNumber++;
            }
	    break;
	    }
	case PARAM_SEGMENT_IPX_IF_DATA:
	    {
	    ipxInterfaceParamsSeen = TRUE;
	    ipxInterface[ipxInterfaceNumber] = paramDataPtr;
	    ipxInterfaceNumber++;
	    break;
	    }
	case PARAM_SEGMENT_IPX_STATIC_ROUTE_DATA:
	    {
	    ipxRouteParamsSeen = TRUE;
	    ipxRoute[ipxRouteNumber] = paramDataPtr;
	    ipxRouteNumber++;
	    break;
	    }
	case PARAM_SEGMENT_IPX_STATIC_SERVICE_DATA:
	    {
	    ipxService[ipxServiceNumber] = paramDataPtr;
	    ipxServiceNumber++;
	    break;
	    }
	case PARAM_SEGMENT_IPX_RIP_FILTER:
	    {
	    ipxFilterParamsSeen = TRUE;
	    ipxRipFilter[ipxRipFilterNumber] = paramDataPtr;
	    ipxRipFilterNumber++;
	    break;
	    }
	case PARAM_SEGMENT_IPX_SAP_FILTER:
	    {
	    ipxFilterParamsSeen = TRUE;
	    ipxSapFilter[ipxSapFilterNumber] = paramDataPtr;
	    ipxSapFilterNumber++;
	    break;
	    }
	case PARAM_SEGMENT_IPX_ADDRESS_FILTER:
	    {
	    ipxFilterParamsSeen = TRUE;
	    ipxAddressFilter[ipxAddressFilterNumber] = paramDataPtr;
	    ipxAddressFilterNumber++;
	    break;
	    }
	case PARAM_SEGMENT_IP_FILTER:
	    {
	    ipFilterParamsSeen = TRUE;
	    ipFilter[ipFilterNumber] = paramDataPtr;
	    ipFilterNumber++;
	    break;
	    }
	    
	case PARAM_SEGMENT_PPP_DATA:
	    {
            if ((HardwareType != ROM_CRATE_HW_MX1608) || (pppPortNumber <= portNumberMX1608)) /* SC65492 FBS 092895 */
            {		
	    pppParamsSeen = TRUE;
	    pppPortTable[pppPortNumber] = paramDataPtr;
	    pppPortNumber++;
	    }
	    break;
	    }
	case PARAM_SEGMENT_CCL_DATA:
	    {
            if ((HardwareType != ROM_CRATE_HW_MX1608) || (cclPortNumber <= portNumberMX1608)) /* SC65492 FBS 092895 */
            {		
	    cclParamsSeen = TRUE;
	    cclPortTable[cclPortNumber] = paramDataPtr;
	    cclPortNumber++;
	    }
	    break;
	    }

	case PARAM_SEGMENT_EXPANDED_PORT_DATA:
	    {
            if ((HardwareType != ROM_CRATE_HW_MX1608) || (expandedPortNumber <= portNumberMX1608)) /* SC65492 FBS 092895 */
            {		
	    expandedPortParamsSeen = TRUE;
	    expandedPortTable[expandedPortNumber] = paramDataPtr;
	    expandedPortNumber++;
	    }
	    break;
	    }

	default:
	    break;
	    }
	paramDataPtr += paramSegmentLength;
	paramSegmentPtr = paramSegmentPtr + PARAM_MSG_SEGMENT_STR_sizelen +
	    paramSegmentLength;
	}
    ProcessKeywordOptions();
    
    }

/*
 ** ProcessKeywordOptions()
 ** 
 ** This routine will process all options that have not been processed yet.
 ** ie. options that do not correspond to a parameter segment.
 ** It calls the option given option processing routine to do the necessary work.
 ** It will check to make sure we do not process duplicate information, if it has
 ** aslready been processed in a parameter segment.
 */
void ProcessKeywordOptions()
{    
    if (displayServer || displayAll)
	{
	ProcessServerOption();
	}
    ProcessFeaturesOption();
    ProcessIpOption();
    ProcessSnmpOption();
    ProcessDaemonOption();
    if (!paramServerParamsSeen)
	{
	/* 
	 ** Option was not printed with Parameter Segment 
	 */
	ProcessParamServerOption();
	}
    if (!securityParamsSeen)
	{
	/* 
	 ** Option was not printed with Parameter Segment 
	 */
	ProcessSecurityOption();
	}

    if (displayPort || displayAll)
	{
	ProcessPortOption();
	}
    if (!kerberosParamsSeen)
	{
	/* 
	 ** Kerberos params have not been printed with parameter segment
	 */
	ProcessKerberosOption();
	}
    if (!securIdParamsSeen)
	{
	/* 
	 ** SecureID params have not been printed with parameter segment
	 */
	ProcessSecurIdOption();
	}
    ProcessArapOption();
    ProcessIpxOption();
    ProcessIpxRouteOption();
    ProcessIpxFilterOption();    
    ProcessIpFilterOption();
    if (!menuParamsSeen)
	{
	/* 
	 ** Menu params have not been printed with parameter segment
	 */
	ProcessMenuOption();
	}
    ProcessSessionOption();
    ProcessLineOption();
    ProcessTelnetOption();
    ProcessSlipOption();
    if (!xremoteParamsSeen)
	{
	/* 
	 ** Option was not printed with Parameter Segment 
	 */
	ProcessXRemoteOption();    
	}
    ProcessPPPOption();
    ProcessRadiusOption();	
}    

/*
 ** ProcessIpxFilterOption()
 */
void ProcessIpxFilterOption()
{
    if (displayIpxFilters && ipxFilterParamsSeen)
	{
	PrintServerIpxRipFilterData();
	PrintServerIpxSapFilterData();
	PrintServerIpxAddressFilterData();
	}
    
    } /* End ProcessIpxFilterOption() */

/*
 ** ProcessIpxRouteOption()
 */
void ProcessIpxRouteOption()
{
    if (displayIpxStaticData && ipxRouteParamsSeen)
	{
	PrintServerIpxStaticRouteData((char *)ipxRoute[0]);
	PrintServerIpxStaticServiceData((char *)ipxService[0]);
	}
    
    } /* End ProcessIpxRouteOption() */

/*
 ** ProcessIpFilterOption()
 */
void ProcessIpFilterOption()
{
    if (displayIpFilters && ipFilterParamsSeen)
	{
	PrintServerIpFilterData();
	}
    } /* End ProcessIpFilterOption() */

/*
 ** ProcessPortOption()
 */

void ProcessPortOption()
{
    GeneratePortInfo( allPortIndexOffsetTable, INDEX_ALL, portBits, displayVerbose);
}

/*
 ** ProcessServerOption()
 */

void ProcessServerOption()
{
    PrintServerParams( serverPtr );
    PrintServerKerberosData( serverPtr );
    PrintServerMenuData( serverPtr );
    PrintServerPPPData( serverPtr );
    PrintServerInternetData( serverPtr );
    /* PJB Added new argument pointer in support of the 8 extra 
     * SNMP gets, sets and traps- V6-0-3 release. 
     */
    PrintServerInternetSnmpData( serverPtr, snmpExpServerPtr );
    PrintServerParameterData( serverPtr );
    PrintServerFeaturesData( serverPtr );
    PrintServerDaemonsData( serverPtr );
    if (arapParamsSeen == TRUE)
	{
	PrintServerArapData( arapServerPtr );
	}
    if ((ipxInterfaceParamsSeen == TRUE) || xprinterEnabled)
	{
	PrintServerIpxData( serverPtr, (char *) ipxInterface[0]);
	}
}


/*
 ** ProcessSlipOption()
 **
 ** This option gives precedence to the All and Port option.
 ** The information is grouped with the Port information if it is
 ** specified.  All is equivalent to specifying the Port option
 */

void ProcessSlipOption()
{
    if (displaySlip && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_SLIP, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessRadiusOption()
 **
 ** This option gives precedence to the All and Port option.
 ** The information is grouped with the Port information if it is
 ** specified.  All is equivalent to specifying the Port option
 */

void ProcessRadiusOption()
{
    if (displayRadius && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_RADIUS, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessSessionOption()
 **
 ** This option gives precedence to the All and Port option.
 ** The information is grouped with the Port information if it is
 ** specified.  All is equivalent to specifying the Port option
 */

void ProcessSessionOption()
{
    if (displaySession && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_SESSION, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessLineOption()
 **
 ** This option gives precedence to the All and Port option.
 ** The information is grouped with the Port information if it is
 ** specified.  All is equivalent to specifying the Port option
 */
void ProcessLineOption()
{
    if (displayLine && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_LINE, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessTelnetOption()
 **
 ** This option gives precedence to the All and Port option.
 ** The information is grouped with the Port information if it is
 ** specified.  All is equivalent to specifying the Port option
 */
void ProcessTelnetOption()
{
    if (displayTelnet && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_TELNET, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessSecurityOption()
 **
 ** This option gives precedence to the All and Port option.
 ** The information is grouped with the Port information if it is
 ** specified.  All is equivalent to specifying the Port option
 */
void ProcessSecurityOption()
{
    if (displaySecurity && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_SECURITY, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessXRemoteOption()
 **
 ** This option gives precedence to the All, Port, and Server option.
 ** The information is grouped with the Port and Server information if 
 ** either of them are specified.  All is equivalent to specifying both.
 */
void ProcessXRemoteOption()
{
    if (displayXRemote && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_XREMOTE, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessSecurIdOption()
 **
 ** This option gives precedence to the All, Port, and Server option.
 ** The information is grouped with the Port and Server information if 
 ** either of them are specified.  All is equivalent to specifying both.
 */
void ProcessSecurIdOption()
{
    if (displaySecurId && !displayAll && !displayPort)
	{
	GeneratePortInfo( allPortIndexOffsetTable, INDEX_SECURID, 
			 portBits, displayVerbose);
	}
}

/*
 ** ProcessDaemonOption()
 **
 ** This option gives precedence to the All and Server option.
 ** The information is grouped with the Server information if it is
 ** specified.  All is equivalent to specifying the Server option
 */
void ProcessDaemonOption()
{
    if (displayDaemons && !displayAll && !displayServer)
	{
	PrintServerDaemonsData( serverPtr );
	}
}

/*
 ** ProcessParamServerOption()
 **
 ** This option gives precedence to the All and Server option.
 ** The information is grouped with the Server information if it is
 ** specified.  All is equivalent to specifying the Server option
 */
void ProcessParamServerOption()
{
    if (displayParamServer && !displayAll && !displayServer)
	{
	PrintServerParameterData ( serverPtr );
	}
}

/*
 ** ProcessSnmpOption()
 **
 ** This option gives precedence to the All and Server option.
 ** The information is grouped with the Server information if it is
 ** specified.  All is equivalent to specifying the Server option
 */
void ProcessSnmpOption()
{
    if (displaySnmp && !displayAll && !displayServer)
	{
	/* PJB Added new argument pointer in support of the 8 extra 
	 * SNMP gets, sets and traps- V6-0-3 release. 
	 */
	PrintServerInternetSnmpData( serverPtr, snmpExpServerPtr );
	}
}

/*
 ** ProcessFeaturesOption()
 **
 ** This option gives precedence to the All and Server option.
 ** The information is grouped with the Server information if it is
 ** specified.  All is equivalent to specifying the Server option
 */
void ProcessFeaturesOption()
{
    if (displayFeatures && !displayAll && !displayServer)
	{
	PrintServerFeaturesData( serverPtr );
	}
}

/*
 ** ProcessIpOption()
 **
 ** This option gives precedence to the All and Server option.
 ** The information is grouped with the Server information if it is
 ** specified.  All is equivalent to specifying the Server option
 */
void ProcessIpOption()
{
    if (displayIp && !displayAll && !displayServer)
	{
	PrintServerInternetData( serverPtr );
	}
}

/*
 ** ProcessPPPOption()
 **
 ** This option gives precedence to the All, Port, and Server option.
 ** The information is grouped with the Port and Server information if 
 ** either of them are specified.  All is equivalent to specifying both.
 */
void ProcessPPPOption()
{
    if (displayPPP && !displayAll)
	{
	if (!displayServer)
	    {
	    PrintServerPPPData( serverPtr);
	    }
	if (!displayPort)
	    {
	    GeneratePortInfo( allPortIndexOffsetTable, INDEX_PPP, 
			     portBits, displayVerbose);
	    }
	}
}

/*
 ** ProcessKerberosOption()
 **
 ** This option gives precedence to the All, Port, and Server option.
 ** The information is grouped with the Port and Server information if 
 ** either of them are specified.  All is equivalent to specifying both.
 */
void ProcessKerberosOption()
{
    if (displayKerberos && !displayAll)
	{
	if (!displayServer)
	    {
	    PrintServerKerberosData( serverPtr );
	    }
	if (!displayPort)
	    {
	    GeneratePortInfo( allPortIndexOffsetTable, INDEX_KERBEROS, 
			     portBits, displayVerbose);
	    }
	}
}

/*
 ** ProcessArapOption()
 **
 ** This option gives precedence to the All, Port, and Server option.
 ** The information is grouped with the Port and Server information if 
 ** either of them are specified.  All is equivalent to specifying both.
 */
void ProcessArapOption()
{
    if (displayArap && !displayAll  && arapParamsSeen)
	{
	if (!displayServer)
	    {
	    PrintServerArapData( arapServerPtr );
	    }
	if (!displayPort)
	    {
	    GeneratePortInfo( arapPortIndexOffsetTable, INDEX_ALL, 
			     portBits, displayVerbose);
	    }
	}
}

/*
 ** ProcessIpxOption()
 **
 ** This option gives precedence to the All, Port, and Server option.
 ** The information is grouped with the Port and Server information if 
 ** either of them are specified.  All is equivalent to specifying both.
 */
void ProcessIpxOption()
{
    if (displayIpx && !displayAll  && ipxInterfaceParamsSeen)
	{
	if (!displayServer)
	    {
	    PrintServerIpxData( serverPtr, (char *)ipxInterface[0] );
	    }
	if (!displayPort)
	    {
	    GeneratePortInfo( ipxIndexOffsetTable, INDEX_ALL, 
			     portBits, displayVerbose);
	    }
	}
}

/*
 ** ProcessMenuOption()
 **
 ** This option gives precedence to the All, Port, and Server option.
 ** The information is grouped with the Port and Server information if 
 ** either of them are specified.  All is equivalent to specifying both.
 */
void ProcessMenuOption()
{
    if (displayMenu && !displayAll)
	{
	if (!displayServer)
	    {
	    PrintServerMenuData( serverPtr );
	    }
	if (!displayPort)
	    {
	GeneratePortInfo( menuPortIndexOffsetTable, INDEX_ALL, 
			 portBits, displayVerbose);
	    }
	}
}

