/* $Id: csportd.c,v 1.31 1997/08/19 19:55:20 pbellino Exp $*/
/*
 ** csportd.c
 **
 ** Multi-purpose Un*x to Comm Server port connectivity tool.
 **
 ** This tool allows connections to be established between a UNIX host and a 
 ** port on a Comm Server.  This tool works has the following major modes:
 **
 ** Print Filter  ("BSD")
 ** ---------------------
 **
 **    This tool will read data from stdin and send it to the Comm Server port.
 **    The user may select LF->LFCR translations and/or EOR reflection.
 **
 ** 2 way communication pipe
 ** ------------------------
 **
 **    This tool will read data from a pseudo terminal or a FIFO (named pipe)
 **    and send it to the Comm Server port.  Likewise, any data read from the
 **    Comm Server port will be sent to the pseudo terminal or FIFO.  In this
 **    mode the user may elect to "drop" any data comming from the Comm Server
 **    port.
 **
 ** Refer to the character strings at "optionsStrs" for all the valid 
 ** switches and combinations.
 **
 ******************************************************************************
 ******************************************************************************
 **
 **     Date     Who   What
 **   ---------+-----+--------------------------------------------------------
 **   05/20/93   GJG   Original, modified from the pty daemon.
 **
 */ 

#include <sys/types.h>
#include <sys/stat.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <fcntl.h>
#include <malloc.h>
#include <stdio.h>
#ifdef NCR
#include <sys/signal.h>
#else
#include <signal.h>
#endif
#include <string.h>
#include <arpa/inet.h>

#include "config.h"

#include <sys/termio.h>

#ifdef PTYSTYLE_SYSV
#include <sys/stropts.h>
#endif

/*
** MIPS has the FD_SET() macros used with select() in bsd/sys/types.h
*/

#ifdef MIPS
#include <bsd/sys/types.h>
#endif 

/* 
** AIX has the FD_SET() macros and other select() stuff in sys/select.h
*/

#ifdef AIX
#include <sys/select.h>
#endif



/*
 ** Interactive Unix has network errnos in net/errno.h
110994 SC37482 SAS  But SCO Unix does not have it there and it does
                    define "i386"!
020895 SC45656 EKC/SAS Same as above but for NCR...
 */

#ifdef i386
#ifndef SCO
#ifndef NCR
#include <net/errno.h>
#endif
#endif
#endif  /* i386 */

/*
 ** HP-UX has pty ioctl stuff in a separate include file
 */

#ifdef HPUX
#include <sys/ptyio.h>
#endif

/*
 ** Software version number
 */

char *softwareVersion = "Version 2.4";

/*
 ** Define possible input/output sources/destinations.
 */

#define  STANDARD_IN       1
#define  STANDARD_OUT      2
#define  PTY               3
#define  PIPE              4
#define  DEV_NULL          5

/*
 ** Declare tuneable constants.
 */
#define BUFFER_MAX      8192    /* size in bytes of a data buffer          */
#define CALLBACK_TIME   30      /* seconds to wait for server to call back */
#define RESPONSE_TIME   10      /* seconds to wait before re-registering   */
#define MAXIMUM_RETRIES 0       /* 0 means try forever                     */
#define IDLE_TIME       5       /* seconds to wait to check pty for data   */

/*
 ** Additional constants.
 */

#ifndef TRUE
#define TRUE 1
#endif   

#ifndef FALSE
#define FALSE 0
#endif

#define REGISTRATION_OK                 0x00
#define REGISTRATION_QUEFULL            0x05
#define MIN_REGISTRATION_RESPONSE_SIZE  4
#define REGISTRATION_RESPONSE_SIZE      6
#define PACKET_MAX                      20
#define NMLEN                           257
#define PRINT_PORT                      173

/*
 ** Constants describing telnetReadState's possible values.
 */

#define TELNET_READ_NORMAL      0
#define TELNET_READ_IAC         1
#define TELNET_READ_DO          2
#define TELNET_READ_WILL        3

/*
 ** Telnet special character definitions.
 */

#define TELNET_IAC_CHAR         0xFF
#define TELNET_DONT_CHAR        0xFE
#define TELNET_DO_CHAR          0xFD
#define TELNET_WONT_CHAR        0xFC
#define TELNET_WILL_CHAR        0xFB
#define TELNET_BREAK_CHAR       0xF3

/*
 ** Macro to return number of bytes remaining in buffer.
 */
#define BUFFER_ROOM(x) (x->endOfBuffer - x->end + 1)

/*
 ** Macro to check for an empty buffer.
 */
#define BUFFER_EMPTY(x) (x->end == x->start)

/*
 ** Macro to reset a buffer.
 */
#define BUFFER_RESET(x) (x->end = x->start = x->buffer)

/*
 ** max() macro.
 */
#ifndef max
#define max(a, b)  ((a > b) ? a : b)
#endif
/*
 ** Packet sturcture.
 */

typedef struct
{
    unsigned int        size;
    int                 index;
    char                buffer[PACKET_MAX];
    } Packet;


/*
 ** Buf structure.
 */

typedef struct
{
    char        *start;
    char        *end;
    char        *endOfBuffer;
    int          maxSize;
    char         buffer[1];  /* will be as large as maxSize, see NewBuf() */
    } Buf;

/*
 ** Config structure.
 */

typedef struct 
{
    int                      debugLevel;
    char                     writeOnlyMode;
    char                     telnetMode;
    char                     gatewayMode;
    char                     permanentMode;
    char                     filterMode;
    char                     eorMode;
    char                     lfcrMode;
    char                     initiateMode;
    char                     breakMode;
    char                     breakChar;
    char                     createPipe;
    char                     createPty;
    char                     ptySymLink;
    char                     ptyUseConfigFile;
    char                     ptyOpostToggle;
    char                     portLogical;
    char                    *ptyConfigFileName;
    char                    *deviceFileName;
    char                     ptyMasterName[NMLEN];
    char                     ptySlaveName[NMLEN];
    int                      userInFd;
    int                      userOutFd;
    int                      commServerFd;
    struct sockaddr_in       serverAddr;
    int                      physicalPort;
    void                    (*userToServerFilter)();
    void                    (*serverToUserFilter)();
    } Config;


/*
 ** External variables.
 */

extern int       errno;
extern int       optind;
extern char     *optarg;

/*
 ** External functions.
 */

extern int        link();

#ifdef HAS_SYMLINK
extern int        symlink();
#else
int symlink(){};
#endif

/*
 ** Global variables.
 */
     
unsigned char       userSource = STANDARD_IN;
unsigned char       userDestination = DEV_NULL;
unsigned char       lostConnection;
Config              config;
Buf                *userInBuf, *toServerBuf, *fromServerBuf, *userOutBuf;
char                errs[NMLEN];
int                 tilde;
     
/*
 ** Usage and Option strings.
 */
    
char *usageStr  = "\nUsage: %s [<options>] <server> <port>\n";
     
char *optionStrs[] = {
"options are:\n",
"    -d <level>      set debug mode to <level> (default is 0 - no debug)\n",
"    -x              translate LF to LFCR (default is Off)\n",
"    -t              limited TELNET mode On (default is Off)\n",
"    -g              gateway mode On (default is Off)\n",
"    -b <break ch>   send a break sequence using a user defined control\n",
" string in hex. example: -b0x1a for <cntl Z> (default is Off)\n",
"    -w              write only mode On (default is Off)\n",
"    -L              port argument is logical\n",
"    -e              EOR reflection (default is Off)\n",
"    -P <pipe-name>  create pipe <pipe-name> for permanent connection I/O\n",
"    -T <pty-name>   create pty <pty-name> for permanent connection I/O\n",
"options which may follow -T <pty-name> or -P <pipe-name> are:\n",
"    -i              initiate connection with Comm Server\n",
"options which may only follow -T <pty-name> are:\n",
#ifdef HAS_SYMLINK
"    -s              make symbolic link for <pty-name>\n",
"    -D              disconnect between jobs (default is Off)\n",
#endif
"    -c <conf-file>  read names of pty devices to use from <conf-file>\n",
"    -o              toggle pty output postprocessing\n",
};
     
/*
 ** Pre-declared routines.
 */
     
void             Log();
void             LogSysError();
void             GetOptions();
void             Usage();
void             PrintConfig();
int              AllocatePTY();
int              AllocatePTYFromConfig();
void             WaitForUserInput();
Buf             *NewBuf();
void             CatchSigs();
int              Connect();
int              RegisterAndConnect();
int              OpenListenSocket();
void             Register();
void             StartTimer();
void             AlarmHandler();
void             PipeHandler();
void             InterruptHandler();
void             CancelTimer();
unsigned short   GetWord();
void             PutChar();
void             PutWord();
void             TelnetOption();
void             ConfigureUserIn();
void             ConfigureUserOut();
void             EorReflection();
void             FilterLfCr();
void             FilterNormToTelnet();
void             FilterTelnetToNorm();
void             FilterCopy();
void             CopyData();


#if !defined(HAS_BZERO)
void             bzero();
#endif

#if !defined(HAS_BCOPY)
void             bcopy();
#endif

/*
 ** main()
 **
 ** Main.
 **
 ** Returns:
 **
 **     Hopefully not.
 */

main(argc, argv)
     int         argc;
     char       *argv[];
{

    unsigned char    needConnection = TRUE;
    struct   linger  lingerer;
    
    /*
     ** Initialize config structure to defaults....
     */

    bzero( (char *)&config, (int)sizeof( config ) );

    config.userToServerFilter = FilterCopy;
    config.serverToUserFilter = FilterCopy;

    config.filterMode = TRUE;

    config.deviceFileName = NULL;  /* 020895 EKC/SAS SC45656 */

    /*
     ** Parse command line options.
     */
    
    GetOptions( argc, argv );
    
    /*
     ** Print out configuration.
     */
    
    PrintConfig();
    
    /*
     ** Configure input source based upon the user specified options.
     */
    
    ConfigureUserIn();
    
    /*
     ** Configure output destination  based upon the user specifed options.
     */
    
    ConfigureUserOut();
    
    
    /*
     ** Create our 4 buffers.
     */
    
    userInBuf     = (Buf *)NewBuf(BUFFER_MAX);
    toServerBuf   = (Buf *)NewBuf(BUFFER_MAX);
    fromServerBuf = (Buf *)NewBuf(BUFFER_MAX);
    userOutBuf    = (Buf *)NewBuf(BUFFER_MAX);
    
    
    /*
     ** Catch SIGALRM with AlarmHandler() and SIGPIPE with PipeHandler().
     */
    
    CatchSigs();
    
    while(TRUE)
	{
	if (needConnection)
	    {
	    /*
	     ** Connect to the server if the user wants us to initiate the 
	     ** connection.  If not, we'll wait for some input from the user
	     ** before regsitering with the server for a connection.
	     */
	    if (config.initiateMode)
		{ 
		if ((config.commServerFd = 
		     Connect( config.serverAddr.sin_addr.s_addr,
			     config.serverAddr.sin_port ))
                    < 0)
                    {
                    Log( "Unable to connect with server.\n" );
                    exit( EXIT_BAD );
                    }
		}
	    else
		{
		
		/*
		 ** Only wait for input from the user if we are not acting like
		 ** a "filter".
		 */
		
		if (!config.filterMode)
		    {
		    WaitForUserInput( config.userInFd );
		    }
		
		/*
		 ** There's data ... establish a connection with the Comm Server 
		 ** port.
		 */
		
		if (config.gatewayMode)
		    {
		    if ((config.commServerFd = 
			 Connect( config.serverAddr.sin_addr.s_addr,
				 htons( PRINT_PORT)))
			< 0)
			{
			Log( "Unable to connect with server.\n" );
			exit( EXIT_BAD );
			}
		    }
		else
		    {
		    if ((config.commServerFd = 
			 RegisterAndConnect(config.serverAddr.sin_addr.s_addr,
					    config.serverAddr.sin_port))
			< 0)
			{
			Log( "Unable to connect with server.\n" );
			exit( EXIT_BAD );
			}
		    }
		}

	    /*
	     ** Set linger on server socket so any data in kernel 
	     ** buffers has a chance of getting to the server when we
	     ** close the connection.
	     */

	    lingerer.l_onoff = 1;
	    lingerer.l_linger = 120;
	    
	    if  (setsockopt( config.commServerFd, SOL_SOCKET, SO_LINGER, 
			    (char *)&lingerer, sizeof( lingerer ) ) 
		 < 0)
		{
		LogSysError("Setsockopt error");
		exit( EXIT_BAD );
		}
	    }
	
	/*
	 ** Shuffle data between the user and the Comm Server port.
	 */

        CopyData();

	/*
	 ** Do TELNET EOR reflection (maybe)
	 */

        if ( (config.eorMode) && (!lostConnection) )
            {
            EorReflection( config.commServerFd );
            }

	/*
	 ** At this point the user has closed down the input stream.  This 
	 ** closure may have been caused by an error.  In any case, we'll clean
	 ** things up and go look for more data.  
	 **
	 **    *  If we are running as a filter, we'll exit.
	 **    *  If a permanent connection is desired, we won't break the
	 **       connection to the Comm Server.
	 ** If we've lost a connection, there may still be data in the user
	 ** buffers.  We don't want to trash this, we need to send it.
	 */

	if ( !lostConnection)
	    {
	    BUFFER_RESET( userInBuf );
	    BUFFER_RESET( userOutBuf );
	    BUFFER_RESET( fromServerBuf );
	    BUFFER_RESET( toServerBuf );
	    }
	
	if (config.filterMode)
	    {
	    shutdown( config.commServerFd, 2 );
	    close( config.commServerFd );
	    exit( EXIT_OK );
	    }

	if (!config.permanentMode || lostConnection)
	    {                     /* SC41094 112294 SAS If lost connection, */
                                  /* clean up before trying to reconnect */
	    shutdown( config.commServerFd, 2 );
	    close (config.commServerFd );
	    }
	else
	    {
	    needConnection = FALSE | lostConnection;
#if 0
/* 110794 SC42376 SAS  Remove this call. We do not want to get stuck here */
/* waiting for the user to start things up when it might never do that. */
	    WaitForUserInput( config.userInFd );
#endif
	    }
	}
    }

/*
** CopyData()
**
** Moves data back and forth between the user and the Comm Server port until
** the user closes the input stream.
**
** Exits:
**
**	If a user read() fails with a non-EIO error.
*/

void CopyData()
{
    int		 nfds = 0, readyfds, error;
    fd_set	 rfds, wfds, efds;

#ifdef HPUX
    struct request_info tioc_req;
#endif
    
#ifdef MIPS
    struct timeval	 timeout;
    
    timeout.tv_sec = IDLE_TIME;
    timeout.tv_usec = 0;
#endif
    
    Log("Copying data.");
    
    nfds = max( nfds, config.userInFd );
    nfds = max( nfds, config.userOutFd );
    nfds = max( nfds, config.commServerFd );
    
    nfds++;
    
    lostConnection = FALSE;
    
    if (config.debugLevel > 2)
	{
	sprintf(errs, 
		"CS Fd  = 0x%02X, User In Fd = 0x%02X, User Out Fd = 0x%02X.",
		config.commServerFd, config.userInFd, config.userOutFd);
	Log(errs);
	}
    
    
    while (TRUE)
	{
	
	
	/*
	 ** If there's data to translate, do it!
	 */
	
	(*config.userToServerFilter)( userInBuf, toServerBuf );
	(*config.serverToUserFilter)( fromServerBuf, userOutBuf );
	
	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	FD_ZERO(&efds);
	
	/*
	 ** Set selection masks. If there is data to write,
	 ** don't look for reads.
	 */
	
	if (!BUFFER_EMPTY( toServerBuf ))
	    {
	    /*
	     ** We have data for the server.
	     */
	    
	    FD_SET(config.commServerFd, &wfds);
	    }
	else
	    {
	    /*
	     ** Look for data from the user.
	     */
	    FD_SET(config.userInFd, &rfds);
#ifdef HPUX
	/* On HP-UX, this allows the master side to see opens and the final close
	** on the slave side.
	*/
	    if (config.createPty) /* 020795 SAS SC46580 Do this only for pty case */
		FD_SET(config.userInFd, &efds);	
#endif
	    }
	
	if (!BUFFER_EMPTY( userOutBuf ))
	    {
	    /*
	     ** We have data for the user.
	     */
	    FD_SET(config.userOutFd, &wfds);
	    }
	else
	    {
	    /*
	     ** Look for data from the server.
	     */
	     FD_SET(config.commServerFd, &rfds);
	    }
	

	if (config.debugLevel > 2)
	    {
	    sprintf(errs,
		    "Selecting: rfds 0x%02X, wfds 0x%02X, efds 0x%02X.",
		    rfds.fds_bits[0], wfds.fds_bits[0], efds.fds_bits[0]);
	    Log(errs);
	    }
	
	/*
	 ** Wait for devices to become ready for I/O.
	 */
	
#ifdef MIPS
	readyfds = select(nfds, &rfds, &wfds, &efds, &timeout);
#else
	readyfds = select(nfds, &rfds, &wfds, &efds, 
			  (struct timeval *)0);
#endif
	
	if (config.debugLevel > 2)
	    {
	    sprintf(errs,
		    "Select(%d): rfds 0x%02X, wfds 0x%02X, efds 0x%02X.",
		    readyfds, rfds.fds_bits[0], wfds.fds_bits[0],
		    efds.fds_bits[0]);
	    Log(errs);
	    }
	
	
	
#ifdef MIPS
	/*
	 ** The MIPS platform doesn't behave quite the same as
	 ** the Sun's.  We have to use a timeout to break out of
	 ** the select() when the pty is idle to try to read the
	 ** pty.  A read failure on the pty means that the client
	 ** has closed the slave end of the pty.  (I think.)
	 */
	
	if (!readyfds && (BUFFER_ROOM(userInBuf)))
	    {
	    FD_SET(config.userInFd, &rfds);
	    }
#endif
    

    /*
     ** Read and write as indicated.
     */
    
    if (FD_ISSET(config.commServerFd, &wfds))
	{
	
	Log("Writing to Comm Server port.");
	
	if (WriteBuf(config.commServerFd, toServerBuf))
	    {
	    Log("Write server error in CopyData()");
	    lostConnection = TRUE;
	    if (config.permanentMode)
		{
		Log( "Assuming connection with server lost" );
		Log( "Will attempt a reconnect" );
		}
	    break;
	    }
	}
    
    if (FD_ISSET(config.commServerFd, &rfds))
	{
	
	Log( "Reading from Comm Server port.");
	
	if (ReadBuf(config.commServerFd, fromServerBuf))
	    {
	    Log("Read server error in CopyData()");
	    lostConnection = TRUE;
	    if (config.permanentMode)
		{
		Log( "Assuming connection with server lost" );
		Log( "Will attempt a reconnect" );
		}
	    break;
	    }
	}
    
    if (FD_ISSET(config.userInFd, &rfds))
	{
	
	Log("Reading from user.");
	
	error = ReadBuf(config.userInFd, userInBuf);
	
	if (error == EIO)
	    {
	    /*
	     ** Error on user input or close...
	     */
	    
	    Log("Read EIO from user device in CopyData().");
/* 112294 SC42376 SAS  Upon removing the WaitForUserInput call in the */
/* main loop, we need to pause here if we read from the user and get  */
/* no data to avoid ending up looping at warp speed. */
	    if ((config.createPty == TRUE) && (BUFFER_EMPTY(userInBuf))
			&& (config.permanentMode))
		{
		StartTimer(IDLE_TIME);
		pause();
		continue;
		}
	    break;
	    }
	else if (error)
	    {
	    /*
	     ** A fatal error has occurred.
	     */
	    
	    Log("Read from user error in CopyData().");
	    exit(EXIT_BAD);
	    }
	}
    
    /*
     ** The check for "write to user" is after the check for "read from user"
     ** for the following reason.  If the user closed down their side of the 
     ** file, then the previous read would have returned to us an EIO error.
     ** Some UNIX's get upset if we try to write to the user after this event.
     ** If an EIO error was returned, then the previous logic would break us out
     ** of the while (TRUE) {} loop and we won't even execute this code.  
     **
     ** Since the user has closed their side of the file, we assume they don't 
     ** want any data we might have for them.
     */
     
    if (FD_ISSET(config.userOutFd, &wfds))
	{
	
	Log("Writing to user.");
	
	if (WriteBuf(config.userOutFd, userOutBuf))
	    {
	    Log("Write user error in CopyData()");
	    break;
	    }
	}

#ifdef HPUX
/* On HP-UX, handle exceptions. TIOREQCHECK is used to determine what
** kind of exception it is (open, close, and maybe ioctl), but the HP
** documentation does not make it clear whether or not ioctls will show
** up. Opens must be "acknowledged" using TIOCREQSET. The pty(7) man
** pages say close does not need to be acknowledged, but in practice,
** it must be, or else csportd will hang when it gets another open.
*/
	if (FD_ISSET(config.userInFd, &efds))
	   {
	   if (ioctl( config.userInFd, TIOCREQCHECK, &tioc_req) < 0)
		{
		LogSysError( "TIOCREQCHECK ioctl failed in CopyData()" );
		exit( EXIT_BAD );
		}
	   else
		{
		if (tioc_req.request == TIOCCLOSE)
		    {
		    Log("Recvd close from user device in CopyData()");
		    if (ioctl( config.userInFd, TIOCREQSET, &tioc_req) < 0)
			{
			LogSysError( "TIOCREQSET ioctl failed for TIOCCLOSE in CopyData()" );
			exit( EXIT_BAD );
			}
		    break;
		    }
		else if (tioc_req.request == TIOCOPEN)
		    {
		    if (config.debugLevel > 2)
			{
			Log("Recvd open from user device in CopyData()");
			}
		    if (ioctl( config.userInFd, TIOCREQSET, &tioc_req) < 0)
			{	
			LogSysError( "TIOCREQSET ioctl failed for TIOCOPEN in CopyData()" );
			exit( EXIT_BAD );
			}
		    }
		else
		    {
		    if (config.debugLevel > 2)
			{
			sprintf(errs, "Recvd unknown trap %d from user device in CopyData()",
			    tioc_req.request);
			Log(errs);
			}
		    if (ioctl( config.userInFd, TIOCREQSET, &tioc_req) < 0)
			{	
			LogSysError( "TIOCREQSET ioctl failed for unknown trap in CopyData()" );
			exit( EXIT_BAD );
			}
		    /* then continue to look for more data... */
		    }
		}
	   }
#endif

    }

    /*
     ** The user has closed their end of the file, but there still may be data
     ** floating around in various buffers.  We need to get this data to the 
     ** Comm Server.  (Only if we still have a connection!)
     */

    if (!lostConnection)
	{
	while ( (!BUFFER_EMPTY( userInBuf )) ||
	       (!BUFFER_EMPTY( toServerBuf )))
	    {
	    (*config.userToServerFilter)( userInBuf, toServerBuf );
	    
	    if (WriteBuf( config.commServerFd, toServerBuf ))
		{
		Log("Write server error in CopyData()");
		break;
		}
	    }
	}

    Log("Finished copying data.");
    }

/*
 ** Log(message)
 **
 ** If the debug option was set, prints out the current timestamp,
 ** program name, and message.
 **
 ** Arguments:
 **
 **     message         The message to be printed.
 */

void
    Log(message)
char    *message;
{
    time_t      t;
    char        *timeStr;
    int         i;
    
    if (!config.debugLevel)
        {
        return;
        }

    t = time(0);
    timeStr = (char *)ctime( &t );

    for (i=0; i<strlen(timeStr); i++)
	{
	if (timeStr[i] == '\n')
	    {
	    timeStr[i] = ':';
	    break;
	    }
	}
    
    fprintf(stderr, "%s %s\n", timeStr, message);


    }

/*
 ** LogSysError(message)
 **
 ** Print out a system error message.
 **
 ** Arguments:
 **
 **     message         The error message to be printed.
 */

void
    LogSysError(message)
char    *message;
{
    time_t      t;
    char        *timeStr;
    int         i;
    
    t = time(0);
    timeStr = (char *)ctime( &t );
    for (i=0; i<strlen(timeStr); i++)
	{
	if (timeStr[i] == '\n')
	    {
	    timeStr[i] = ':';
	    break;
	    }
	}

    fprintf(stderr, "%s ", timeStr);
    perror(message);
    }

/*
 ** GetOptions(argc, argv )
 **
 ** Parses the command-line arguments specified in argc/argv into
 ** the Config structure pointed to by c.
 **
 ** Arguments:
 **
 **     argc            The number of arguments in argv.
 **
 **     argv            A list of argument strings.
 **
 */

void
    GetOptions(argc, argv )
int       argc;
char    **argv;
{
    
#ifdef HAS_SYMLINK
    char   *options = "d:fxewtb:gP:T:sic:oDL";
#else
    char   *options = "d:fxewtb:gP:T:ic:oDL";
#endif
    
    int          opt;
    char                *appName;
    struct hostent      *hostEnt;
    
    appName = *argv;
    
    while ((opt = getopt(argc, argv, options )) != -1)
        {
        switch (opt)
            {
        case 'd':
            {
            config.debugLevel = atoi(optarg);
            break;
            }
            
        case 'x':
            {
            config.lfcrMode = TRUE;
	    config.userToServerFilter = FilterLfCr;
	    config.serverToUserFilter = FilterCopy;
            break;
            }
            
        case 'e':
            {
            if (config.ptyOpostToggle)
                {
                Usage( "Options -e and -o are incompatible", appName);
                }

	    if (config.createPipe)
                {
                Usage("Options -P and -e incompatible",
                      appName);
                }
            config.eorMode = TRUE;
            break;
            }
            
        case 'w':
            {
            config.writeOnlyMode = TRUE;
            userDestination = DEV_NULL;
            break;
            }
            
        case 't':
            {
            config.telnetMode = TRUE;
	    config.userToServerFilter = FilterNormToTelnet;
	    config.serverToUserFilter = FilterTelnetToNorm;
            break;
            }
            
        case 'g':
            {
            config.gatewayMode = TRUE;
            break;
            }
            
        case 'P':
            {
            if (config.createPty)
                {
                Usage("Options -P and -T incompatible",
                      appName);
                }
            config.createPipe = TRUE;
            config.deviceFileName = optarg;
            config.permanentMode = TRUE;
            config.filterMode = FALSE;
            userSource      = PIPE;
            userDestination = PIPE;
            break;
            }
            
        case 'T':
            {
            if (config.createPipe)
                {
                Usage("Options -P and -T incompatible",
                      appName);
                }
            config.createPty = TRUE;
            config.permanentMode = TRUE;
			config.deviceFileName = optarg;
			if (strcmp("/dev", optarg) == 0)
			{
				Usage("/dev invalid, must be in the form of /dev/<device name>.",
					  appName);
			}
            config.filterMode = FALSE;
            userSource       = PTY;
            userDestination = PTY;
            break;
            }
		case 'b':
            {
            if (!config.telnetMode)
			    {
                Usage("Option -b must follow -t", appName);
			    }
			config.breakChar = strtol(optarg, 0, 16);
            config.breakMode = TRUE;
            config.userToServerFilter = FilterNormToTelnet;
            break;
		    }

        case 'i':
            {
            config.initiateMode  = TRUE;
            config.permanentMode = TRUE;
            break;
            }
            
#ifdef HAS_SYMLINK          
        case 's':
            {
            if (!config.createPty)
                {
                Usage("Option -s must follow -T",
                      appName);
                }
            config.ptySymLink = TRUE;
            break;
            }
#endif
            
        case 'c':
            {
            if (!config.createPty)
                {
                Usage("Option -c must follow -T",
                      appName);
                }
            config.ptyUseConfigFile = TRUE;
            config.ptyConfigFileName = optarg;
            break;
            }
            
        case 'o':
            {
            if (!config.createPty)
                {
                Usage("Option -o must follow -T", appName);
                }

            if (config.eorMode)
                {
                Usage( "Options -o and -e are incompatible", appName);
                }

            if (!config.permanentMode)
                {
                Usage( "Options -o and -D are incompatible", appName );
                }
            
            config.ptyOpostToggle = TRUE;
            break;
            }
            
        case 'D':
            {
            if (!config.createPty)
                {
                Usage("Option -D must follow -T", appName);
                }
            
            if (config.ptyOpostToggle)
                {
                Usage( "Options -D and -o are incompatible", appName );
                }
            
            config.permanentMode = FALSE;
            break;
            }
            
        case 'L':
            {
            config.portLogical = TRUE;
            break;
            }
            
        case '?':
            {
            Usage((char *)0, appName);
            }
            }
        }
    
    /*
     *  Now, we expect to have two arguments left.  The server name/IP address and
     *  and the server port number.  
     */
    
    if ((argc - optind) > 2)
        {
        Usage( "Too many arguments.", appName );
        }
    else
        {
        if ((argc - optind) < 2)
            {
            Usage( "Too few arguments.", appName);
            }
        }
    
    /*
     *  Get the server internet address from the command line.
     */
    
    config.serverAddr.sin_addr.s_addr = (int)inet_addr( argv[optind] ); 
    
    if (config.serverAddr.sin_addr.s_addr == -1)
        {
        /*
         ** Argument not in dot notation, try
         ** name lookup.
         */
        
        if ((hostEnt = gethostbyname(argv[optind])) ==
            (struct hostent *)0)
            {
            Usage("Bad name or internet address.",
                  appName);
            }
        
        /*
         ** Copy the address into serverInternetAddress.
         */
        
        bcopy((char *)hostEnt->h_addr,
              (char *)&config.serverAddr.sin_addr.s_addr,
              (int)hostEnt->h_length);
        }
    
    /*
     *  Get the server port from the command line.
     */

    if (config.portLogical)
        {
        config.serverAddr.sin_port = htons( atoi( argv[optind+1] ));
        }
    else
        {
        config.physicalPort = atoi( argv[optind+1] );
        config.serverAddr.sin_port = htons( 2000 + (config.physicalPort * 100));
        }

    /*
     *  Set the family to internet.
     */

    config.serverAddr.sin_family = AF_INET;
    
    }

/*
 ** Usage()
 **
 ** Prints out the an informational message and the usage string
 ** for this process, then calls exit.
 **
 ** Arguments:
 **
 **     message         The informational message to be printed before
 **
 **     appName         The name the program was invoked with.
 **
 ** Exits:
 **
 **     Every time.
 */

void
    Usage(message, appName)
char    *message;
char    *appName;
{
    int numOptStrs, j;
    
    if (message)
        {
        fprintf(stderr, "\n%s \n", message);
        }
    
    fprintf(stderr, "\ncsportd %s\n", softwareVersion);
    fprintf(stderr, usageStr, appName);
    
    numOptStrs = sizeof(optionStrs) / sizeof(optionStrs[0]);
    
    for (j = 0; j < numOptStrs; j++)
        {
        fprintf(stderr, optionStrs[j]);
        }
    
    exit(EXIT_BAD);
    }

/*
 ** PrintConfig()
 **
 ** If the debug option was set, prints out the current configuration.
 */

void PrintConfig( )
     
{
    if (!config.debugLevel)
        {
        return;
        }
    
    fprintf( stderr, "\n" );
    
    if (config.ptyUseConfigFile)
        {
        fprintf( stderr, "        PTY config file: %s\n",
                config.ptyConfigFileName);
        }
    
    fprintf( stderr, "Server Internet Address: %s\n", 
            inet_ntoa(config.serverAddr.sin_addr));    
    
    if (config.gatewayMode)
        {
        fprintf( stderr, "           Gateway Port: %d\n", PRINT_PORT);
        }
    else
        {
        fprintf( stderr, "   Server Internet Port: %d\n", 
                config.serverAddr.sin_port);
        if (!config.portLogical)
            {
            fprintf( stderr, "   Server Physical Port: %d\n", config.physicalPort);
            }
        }
    
    if (config.deviceFileName)
        {
        fprintf( stderr, "            User Device: %s\n\n", 
                config.deviceFileName);
        }

    fprintf( stderr, "Options:\n" );
    
    fprintf( stderr, "  Debug Level : %d\n", config.debugLevel );
    if (config.initiateMode)
        {
        fprintf( stderr, "  Initiate Connection\n" );
        }
    if (config.writeOnlyMode)
        {
        fprintf( stderr, "  Write Only Mode\n" );
        }
    if (config.telnetMode)
        {
        fprintf( stderr, "  TELNET Mode\n" );
        }
    if (config.gatewayMode)
        {
        fprintf( stderr, "  GateWay Mode\n" );
        }
    if (config.permanentMode)
        {
        fprintf( stderr, "  Permanent Connection\n" );
        }
    if (config.filterMode)
        {
        fprintf( stderr, "  Filter Mode\n" );
        }
    if (config.eorMode)
        {
        fprintf( stderr, "  Use TELNET EOR\n" );
        }
    if (config.lfcrMode)
        {
        fprintf( stderr, "  Translate LF to LFCR\n" );
        }
    if (config.createPipe)
        {
        fprintf( stderr, "  Create a pipe\n" );
        }
    if (config.createPty)
        {
        fprintf( stderr, "  Use a pseudo-terminal\n" );
        }
    if (config.ptySymLink)
        {
        fprintf( stderr, "  Create a symbolic link to pseudo-terminal\n" );
        }
    if (config.ptyOpostToggle)
        {
        fprintf( stderr, "  Toggle PTY slave output processing\n" );
        }
    }

/*
 ** AllocatePTY()
 **
 ** Searches through the set of controller ptys trying to open them,
 ** until a free one is found.  Returns the fd representing the PTY
 ** to the caller or a -1 if it can't figure it out.
 */

int  AllocatePTY()
{
    int         bank, unit, length, ptyFd, slaveFd;
    char        ptyName[NMLEN], *slaveName;

    struct termio   ttyMode;        
    struct stat     statBlock;

    /*
     ** If a configuration file was specified on the command line,
     ** try to open the pty devices specified there, instead of the
     ** using the search algorithm below.
     */

    if (config.ptyUseConfigFile)
        {
	if ( (ptyFd = AllocatePTYFromConfig( config.ptyConfigFileName )) < 0)
	    {
	    Log( "Unable to allocate PTY device from config file." );
	    ptyFd = -1;
	    }
	return( ptyFd );
        }
    
#ifdef PTYSTYLE_BSD
    
    strcpy( ptyName, "/dev/ptyp0" );
    length = strlen( ptyName );
    
    for (bank = 0; bank <= strlen(PTYBANKS); bank++)
        {
        ptyName[length - 2] = PTYBANKS[bank];
        ptyName[length - 1] = '0';
        
#ifndef AIX
        /* 012893 SAS - this test doesn't work on IBM RS/6000 */
        if (stat( ptyName, &statBlock) < 0)
            {
            break;
            }
#endif
        
        for (unit = 0; unit < strlen(PTYUNITS); unit++)
            {
            ptyName[length - 1] = PTYUNITS[unit];
            
            sprintf(errs, "Trying to allocate %s.", ptyName);
            Log(errs);
            
            if ((ptyFd = open(ptyName, O_RDWR )) >= 0)
                {
                strcpy( config.ptyMasterName, ptyName );
                strcpy( config.ptySlaveName,  ptyName );
                config.ptySlaveName[strlen( "/dev/" )] = 't';
                sprintf( errs, "PTY Master : %s", config.ptyMasterName );
                Log( errs );
                sprintf( errs, "PTY Slave  : %s", config.ptySlaveName );
                Log( errs );
		/*
		 ** Toggle PTY slave output processing.  On BSD systems, we'll
		 ** disable output processing and place the TTY in "raw" mode.
		 */
		if (config.ptyOpostToggle)
		    {
		    if ((slaveFd = open( config.ptySlaveName, O_RDWR )) < 0)
			{
			LogSysError( "Unable to open slave side of PTY" );
			}
		    else
			{
			if (ioctl( slaveFd, TCGETA, &ttyMode) < 0)
			    {
			    LogSysError( "Unable to ioctl(TCGETS)" );
			    }
			else
			    {

			    ttyMode.c_oflag &= ~OPOST;
			    ttyMode.c_iflag  = 0;
			    ttyMode.c_lflag &= ~(ISIG | ICANON | ECHO | XCASE);
			    ttyMode.c_cflag &= ~(CSIZE | PARENB);
			    ttyMode.c_cflag |= CS8;
			    ttyMode.c_cc[VMIN]  = 1;
			    ttyMode.c_cc[VTIME] = 0;
			    
			    if (ioctl( slaveFd, TCSETA, &ttyMode) < 0)
				{
				LogSysError( "Unable to ioctl(TSCSETA)" );
				}
			    }
			}
		    }

                return( ptyFd );
                }
            }
        }
    
    Log("All PTYs in use.");
    return( -1 );
#endif

#ifdef PTYSTYLE_SYSV

    sprintf( config.ptyMasterName, "/dev/ptmx");
    
    if ( (ptyFd = open( config.ptyMasterName, O_RDWR )) < 0)
        {
        LogSysError( "Unable to open Master PTY device" );
        exit( EXIT_BAD );
        }

    if ( grantpt( ptyFd) < 0 )
        {       
        LogSysError( "Unable to grant a slave PTY device" );
        exit( EXIT_BAD );
        }
    
    if ( unlockpt( ptyFd ) < 0 )
        {       
        LogSysError( "Unable to unlock slave PTY device" );
        exit( EXIT_BAD );
        }

    if ( (slaveName = (char *)ptsname( ptyFd)) == NULL )
        {
        LogSysError( "Unable to allocate slave PTY device" );
        exit( EXIT_BAD );
        }
    
    strcpy( config.ptySlaveName, slaveName );
    sprintf( errs, "PTY Master : %s", config.ptyMasterName );
    Log( errs );
    sprintf( errs, "PTY Slave  : %s", config.ptySlaveName );
    Log( errs );

    /*
     ** On System V systems, the slave device has a file access that won't let
     ** the common user use it.  Let's change it!
     */

    if ( (chmod( config.ptySlaveName, 0666)) < 0)
	{
	LogSysError( "Unable to modify Slave access" );
	}

    /*
     ** Toggle PTY slave output processing.  On Sys V systems, we'll
     ** enable output processing and configure the PTY to act like a TTY.
     */

    if (config.ptyOpostToggle)
	{
	if ((slaveFd = open( config.ptySlaveName, O_RDWR )) < 0)
	    {
	    LogSysError( "Unable to open slave side of PTY" );
	    }

	else
	    {
	    if ( ioctl(slaveFd, I_PUSH, "ptem") < 0 )
		{
		LogSysError( "Unable to push 'ptem' module" );
		}
	    
	    if ( ioctl(slaveFd, I_PUSH, "ldterm") < 0 )
		{
		LogSysError( "Unable to push 'ldterm' module" );
		}
	    }
	}
    return( ptyFd );

#endif
    
    }

/*
 ** AllocatePTYFromConfig()
 **
 ** Searches through the set of controller ptys specified in the
 ** configuration file configFile trying to open them, until a free
 ** one is found.  The global variable ptyName is set to the file
 ** name of the allocated pty.
 **
 ** Exits:
 **
 **     If config file could not be opened or all configured PTYs
 **     are in use.
 */

int   AllocatePTYFromConfig( configFile )
     char *configFile;
{
    FILE        *cf;
    char        ptyName[NMLEN], *slaveName;
    int         pty;
    int		i,j; /* 092894 SAS SC39150 */
    

    /*
     ** Open the pty configuration file.
     */
    
    if ((cf = fopen(configFile, "r")) == (FILE *)0)
        {
        sprintf(errs, "Error opening config file \"%s\"", configFile);
        LogSysError(errs);
        exit( EXIT_BAD );
        }
    
    /*
     ** Read the pty configuration file one line at a time into
     ** ptyName, trying to open the specified pty.
     */
    
    while (fgets(ptyName, NMLEN, cf))
        {
        if (ptyName[0] == '#')
            {
            continue;
            }
        
        ptyName[strlen(ptyName) - 1] = '\0';
        sprintf(errs, "Trying to allocate %s.", ptyName);
        Log(errs);

        if ((pty = open(ptyName, O_RDWR)) >= 0)
            {
	    strcpy( config.ptyMasterName, ptyName );
	    strcpy( config.ptySlaveName,  ptyName );
	    /* 092894 SAS SC39150 Find the first char after the rightmost "/" */
	    /*                    in the slave name and change it to a "t".   */
	    j = 0;
	    for (i = strlen(config.ptySlaveName) - 1; i >= 0; i--)
		{
		if (config.ptySlaveName[i] == '/')
		    {
			j = i + 1;
			break;
		    }
		}
	    config.ptySlaveName[j] = 't';
	    sprintf( errs, "PTY Master : %s", config.ptyMasterName );
	    Log( errs );
	    sprintf( errs, "PTY Slave  : %s", config.ptySlaveName );
	    Log( errs );
	    return( pty );
            }
        }
    /*
     ** There were no available ptys.
     */
    
    Log("All PTYs in use.");
    return( -1 );
    }

/*
 ** NewBuf(size)
 **
 ** Allocates space for a buf of size "size" with malloc().
 **
 ** Arguments:
 **
 **     size            The size to make the buffer inside the buf.
 **
 ** Return value:
 **
 **     A pointer to the newly allocated buf structure.
 **
 ** Exits:
 **
 **     If malloc() fails.
 */

Buf *
    NewBuf(size)
int     size;
{
    Buf *buf;
    
    buf = (Buf *)malloc(sizeof(Buf) + size);
    
    if (buf == (Buf *)NULL)
        {
        Log("Unable to allocate Buf.");
        exit( EXIT_BAD );
        }
    
    buf->maxSize = size;
    buf->start = buf->end = buf->buffer;
    buf->endOfBuffer = buf->start + size - 1;
    return(buf);
    }

/*
 ** CatchSigs()
 **
 ** Uses sigvec() to catch SIGALRM so it will interrupt recvfrom().
 **
 ** Exits:
 **
 **     If sigvec() fails.
 */

void
    CatchSigs()
{


#ifndef SV_INTERRUPT
    
    signal(SIGPIPE, PipeHandler);
    
#else
    
    struct sigvec       vec;

    if (sigvec(SIGALRM, (struct sigvec *)0, &vec) < 0)
        {
        LogSysError("Error loading vector for SIGALRM");
        exit( EXIT_BAD );
        }
    
    vec.sv_handler = AlarmHandler;
    vec.sv_flags |= SV_INTERRUPT;
    
    if (sigvec(SIGALRM, &vec, (struct sigvec *)0) < 0)
        {
        LogSysError("Error storing vector for SIGALRM");
        exit( EXIT_BAD );
        }
    
    if (sigvec(SIGPIPE, (struct sigvec *)0, &vec) < 0)
        {
        LogSysError("Error loading vector for SIGPIPE");
        exit( EXIT_BAD );
        }
    
    vec.sv_handler = PipeHandler;
    vec.sv_flags |= SV_INTERRUPT;
    
    if (sigvec(SIGPIPE, &vec, (struct sigvec *)0) < 0)
        {
        LogSysError("Error storing vector for SIGPIPE");
        exit( EXIT_BAD );
        }

    if (sigvec(SIGINT, (struct sigvec *)0, &vec) < 0)
        {
        LogSysError("Error loading vector for SIGINT");
        exit( EXIT_BAD );
        }
    
    vec.sv_handler = InterruptHandler;
    vec.sv_flags |= SV_INTERRUPT;
    
    if (sigvec(SIGINT, &vec, (struct sigvec *)0) < 0)
        {
        LogSysError("Error storing vector for SIGINT");
        exit( EXIT_BAD );
        }
    
/* 020895 EKC/SAS SC45656 catch SIGTERM and treat it the same as SIGINT */

    if (sigvec(SIGTERM, (struct sigvec *)0, &vec) < 0)
        {
        LogSysError("Error loading vector for SIGTERM");
        exit( EXIT_BAD );
        }
    
    vec.sv_handler = InterruptHandler;
    vec.sv_flags |= SV_INTERRUPT;
    
    if (sigvec(SIGTERM, &vec, (struct sigvec *)0) < 0)
        {
        LogSysError("Error storing vector for SIGTERM");
        exit( EXIT_BAD );
        }
    

#endif
    
    }


 
/*
** WaitForUserInput()
**
** Loops calling select and checking to see if there is data
** to be read from the user.
**
** Exits:
**
**      If select() fails.
*/

void
WaitForUserInput( fd )
     int  fd;
{
    int            error, readyfds;
    fd_set         rfds, wfds, efds;
#ifdef HPUX
    struct request_info tioc_req;
#endif
    struct timeval timeout;

    timeout.tv_sec = IDLE_TIME;
    timeout.tv_usec = 0;

    /*
     ** If there is data already in the buffer, then just return.  This will
     ** happen when we lose our connection with the Comm Server during a data
     ** transfer.  If we are in permanent mode, then this routine might get 
     ** called twice.  
     ** It is also possible that we have read the data from the user and
     ** do it ready to go to the server.
     */
    
    if ((!BUFFER_EMPTY( userInBuf)) || (!BUFFER_EMPTY( toServerBuf)))
	{
	return;
	}
    
    Log("Waiting for input from user.");

    while (TRUE)
        {
        FD_ZERO(&rfds);
        FD_ZERO(&wfds);
        FD_ZERO(&efds);
        
        FD_SET(fd, &rfds);
#ifdef HPUX
    /* On HP-UX, this allows the master side to see opens and the final close
    ** on the slave side.
    */
	if (config.createPty) /* 020795 SAS SC46580 Do this only for pty case */
	    FD_SET(fd, &efds);	
#endif
      do_select:
        if (readyfds = select((fd + 1), &rfds, &wfds, &efds,
                              &timeout) < 0)
            {
            LogSysError("Select error in WaitForUserInput()");
            exit( EXIT_BAD );
            }

        if (config.debugLevel > 2)
            {
            sprintf(errs,
                    "Select(%d): rfds 0x%02X, wfds 0x%02X, efds 0x%02X.",
                    readyfds, rfds.fds_bits[0], wfds.fds_bits[0],
                    efds.fds_bits[0]);
            Log(errs);
            }

#ifdef HPUX
	/* 
	** On HP-UX, handle exceptions. We are expecting only an open here.
	*/
	if (FD_ISSET(fd, &efds))
	   {
	   if (ioctl( fd, TIOCREQCHECK, &tioc_req) < 0)
		{
		LogSysError( "TIOCREQCHECK ioctl failed in WaitForUserInput()" );
		exit( EXIT_BAD );
		}
	   else
		{
		if (tioc_req.request == TIOCOPEN)
		    {
		    Log("Recvd open from user device in WaitForUserInput()");
		    if (ioctl( fd, TIOCREQSET, &tioc_req) < 0)
			{
			LogSysError( "TIOCREQSET ioctl failed for TIOCOPEN in WaitForUserInput()" );
			exit( EXIT_BAD );
			}
		    }
		else if (tioc_req.request == TIOCCLOSE)
		    {
		    Log("Recvd close from user device in WaitForUserInput()");
		    if (ioctl( fd, TIOCREQSET, &tioc_req) < 0)
			{
			LogSysError( "TIOCREQSET ioctl failed for TIOCCLOSE in WaitForUserInput()");
			exit( EXIT_BAD );
			}
		    goto do_select;
		    }
		else
		    {
		    sprintf(errs, "Recvd trap %d from user device in WaitForUserInput()",
			    tioc_req.request);
		    Log(errs);
		    if (ioctl( fd, TIOCREQSET, &tioc_req) < 0)
			{
			LogSysError( "TIOCREQSET ioctl failed for unknown trap in WaitForUserInput()");
			exit( EXIT_BAD );
			}
		    goto do_select;
		    }
		}
	   }
#endif

        /*
         ** We need to make sure data is actually available from the
         ** user instead of just assuming select() setting rfds
         ** means there's data.
         */

        if (FD_ISSET(fd, &rfds))
            {
            if (error = ReadBuf( fd, userInBuf ))
                {
                if (error == EIO)
                    {
                    StartTimer(IDLE_TIME);
                    pause();
                    continue;
                    }
                
                Log("Exiting WaitForUserInput().");
                exit( EXIT_BAD );
                }
            /*
             ** There's data for the server, time to return...
             */
            
            return;
            }
        }
    }


/*
 ** Connect()
 **
 ** Opens a socket and connects directly to the server.  This is used to
 ** print to a LAT printer via TCP through the TCP/LAT gateway or when the
 ** the user wants the UNIX host to initiate the connection.
 **
 ** Arguments:
 **     
 **     serverAddr:    IP address of the Comm Server.
 **
 **     serverPort:    TCP port to connect to.  (On the Comm Server)
 **
 ** Returns:
 **
 **     fd which points to Comm Server port.
 **
 ** Exits:
 **
 **     If socket() fails for a reason besides a timeout.
 */

int Connect( serverAddr, serverPort )
     int              serverAddr;
     unsigned short   serverPort;
     
{
    struct sockaddr_in  addr;
    int                 serverSocket;
    
    addr.sin_family      = AF_INET;
    addr.sin_addr.s_addr = serverAddr;
    addr.sin_port        = serverPort;
    
    Log("Connecting to server port.");

    serverSocket = -1;

    do
        {
        if (serverSocket >= 0)
            {
            LogSysError( "Connect error in Connect(), waiting" );
            close( serverSocket );
            StartTimer( CALLBACK_TIME );
            pause();
            }
        
        /*
         ** Open the server socket.
         */
        
        if ((serverSocket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
            {
            LogSysError ("Socket error in Connect()");
            exit( EXIT_BAD );
            }
        
        /*
         ** Try to connect to the server.
         */
        
        } while (connect(serverSocket, (struct sockaddr *)&addr, 
                         sizeof(addr)) < 0);

    return( serverSocket );
    }

/*
 ** RegisterAndConnect()
 **
 ** Opens a listening TCP socket, registers with the server, and waits
 ** for the server to connect back.
 **
 ** Arguments:
 **     
 **     serverAddr:    IP address of the Comm Server.
 **
 **     serverPort:    TCP port to connect to.  (On the Comm Server)
 **
 ** Exits:
 **
 **     If accept() fails for a reason besides a timeout, or if
 **     OpenListenSocket() or Register() exit.
 **
 ** Returns:
 **
 **     A file descriptor representing the connection to the Comm
 **     Server.
 */

int RegisterAndConnect( serverAddr, serverPort )
     int              serverAddr;
     unsigned short   serverPort;
{
    int                  listenSocket;
    struct sockaddr_in   listenAddr;
    int                  serverSocket;
    
    /*
     ** Establish a port for the callback connection from the server.
     */
    
    
    bzero((char *)&listenAddr, (int)sizeof(listenAddr));
    
    listenSocket = OpenListenSocket( &listenAddr );
    
    /*
     ** Wait for connect callback from the server or the callback
     ** timeout.  If connect callback does not complete in a
     ** reasonable period of time, re-register with the server.
     ** The callback might not complete within the specified time
     ** if either the server has crashed and restarted or the list
     ** of jobs takes a significant time to print. Re-registration
     ** in the former case is most desirable and harmless in the
     ** latter case.
     */
    
    while (1)
        {
        /*
         ** Register with the server.
         */
        
        Register(listenAddr.sin_port, serverAddr, serverPort);
        
        Log("Waiting for connection from server port.");
        
        StartTimer(CALLBACK_TIME);
        
        if ((serverSocket = accept(listenSocket, 0, 0)) >= 0)
            {
            /*
             ** We're connected to the server.
             */
            
            Log("Connected to the server.");
            
            /*
             ** Set the socket to non-blocking IO mode.
             */
#if 0
            if (fcntl(serverSocket, F_SETFL, O_NDELAY) < 0)
                {
                LogSysError("Fcntl error in RegisterAndConnect()");
                }
#endif
            CancelTimer();
            close(listenSocket);
            return( serverSocket );
            }
        
        if (errno != EINTR)
            {
            LogSysError("Accept error in RegisterAndConnect()");
            exit( EXIT_BAD );
            }
        
        /*
         ** The timer fired, re-register.
         */
        
        Log("Re-registering with the server.");
        }
    }

/*
 ** OpenListenSocket()
 **
 ** Opens a TCP socket, binds it to any port number, and listens on it.
 **
 ** Return value:
 **
 **     The bound, listening TCP socket descriptor.
 **
 ** Exits:
 **
 **     If any of: socket(), bind(), getsockname(), or listen() fail.
 */

int OpenListenSocket( addr )
     struct  sockaddr_in  *addr;
{
    int  s, nameLen;
    
    Log("Opening listening socket.");
    
    if ((s = socket(AF_INET,SOCK_STREAM,0)) < 0 )
        {
        LogSysError("Socket error in OpenListenSocket()");
        exit( EXIT_BAD );
        }
    
    /*
     **  Configure our listen address to accept connections from
     **  anybody.  By setting the port (sin_port) to 0, the system will
     **  assign an unused port number for us.
     */
    
    addr->sin_family = AF_INET;
    addr->sin_addr.s_addr = INADDR_ANY;
    addr->sin_port = 0;
    nameLen = sizeof(struct sockaddr_in);
    
    if (bind(s, (struct sockaddr *)addr, nameLen) < 0)
        {
        LogSysError("Bind error in OpenListenSocket()");
        exit( EXIT_BAD );
        }
    
    if (getsockname(s, (struct sockaddr *)addr, &nameLen) < 0)
        {
        LogSysError("Getsockname error in OpenListenSocket()");
        exit( EXIT_BAD );
        }
    
    if (listen(s, 1) < 0)
        {
        LogSysError("Listen error in OpenListenSocket()");
        exit( EXIT_BAD );
        }
    
    sprintf(errs, "Listening on local TCP port %d.", addr->sin_port);
    Log(errs);
    
    return(s);
    }

/*
 ** Register()
 **
 ** Registers a request for callback with the terminal server.
 ** Sends a registration message every RESPONSE_TIME seconds until a
 ** response is received or an error occurs.  If the response is "ok",
 ** registration is complete.  If the response is "registration queue
 ** full", try again.  Exit on other responses.
 **
 ** Arguments:
 **
 **     TCPport         The TCP port number being listened on awaiting
 **                     callback from the server.
 **
 **     serverAddress   IP Address of the Comm Server.
 **
 ** Exits:
 **
 **     If any of socket(), sendto(), recvfrom() fail, or if the server
 **     returns an error result code, or we were unable to register
 **     after MAXIMUM_RETRIES.
 */

void Register( TCPport, serverAddress, serverPortNumber )
     unsigned short     TCPport;
     int                serverAddress;
     unsigned short     serverPortNumber;
     
     
{
    int                 regSocket;
    struct sockaddr_in  regAddr;
    int                 regAddrLength;
    Packet                      regRequest;
    Packet                      regResponse;
    unsigned long               retryCount;
    int                 nbytes, dummy;
    unsigned short              result;
    
    /*
     ** Open a UDP socket.
     */
    
    Log("Opening UDP registration socket.");
    
    if ((regSocket = socket(AF_INET, SOCK_DGRAM, 0)) < 0) 
        {
        LogSysError("Socket error in Register");
        exit( EXIT_BAD );
        }
    
    /*
     ** Set the server's address in regAddr.
     */
    
    bzero((char *)&regAddr, sizeof(regAddr));
    regAddr.sin_family = AF_INET;
    regAddr.sin_addr.s_addr = serverAddress;
    regAddr.sin_port = htons(PRINT_PORT);
    
    /*
     ** Build the registration message.
     */
    
    regRequest.index = 0;
    regRequest.size = 0;
    
    PutChar(&regRequest, 1);            /* our protocol type */
    PutChar(&regRequest, 0);            /* pad */
    PutWord(&regRequest, serverPortNumber);
    PutWord(&regRequest, TCPport);
    PutWord(&regRequest, 0);
    
    /*
     ** Loop MAXIMUM_RETRIES times, (or forever if MAXIMUM_RETRIES is 0)
     ** trying to register with the server.
     */
    
    for (retryCount = 0;
         ((MAXIMUM_RETRIES == 0) || (retryCount < MAXIMUM_RETRIES));
         retryCount++)
        {
        Log("Sending registration request.");
        
        if (sendto(regSocket, regRequest.buffer, regRequest.size, 0,
                   (struct sockaddr *)&regAddr, sizeof(regAddr)) < 0)
            {
            close(regSocket);
            LogSysError("Sendto error in Register().");
            exit( EXIT_BAD );
            }
        
        regResponse.size = REGISTRATION_RESPONSE_SIZE;
        regResponse.index = 0;
        regAddrLength = sizeof(regAddr);
        
        sprintf(errs, "Waiting %d seconds for response.",
                RESPONSE_TIME);
        Log(errs);
        
        StartTimer(RESPONSE_TIME);
        
        if ((nbytes = recvfrom(regSocket, regResponse.buffer,
                               regResponse.size, 0,
                               (struct sockaddr *)0, &dummy)) < 0)
            {
            if (errno == EINTR)
                {
                /*
                 ** The timer fired.
                 */
                
                sprintf(errs, "Request %d timed out.",
                        retryCount);
                Log(errs);
                continue;
                }
            
            LogSysError("Recvfrom error in Register()");
            exit( EXIT_BAD );
            }
        
        /*
         ** If the response message is too short, ignore it.
         **
         ** Compare nbytes to MIN_REGISTRATION_RESPONSE_SIZE
         ** because v3.0 of TS code and earlier return only
         ** 4-byte responses.
         */
        
        if (nbytes < MIN_REGISTRATION_RESPONSE_SIZE)
            {
            sprintf(errs,
                    "Short response to request %d: %d bytes.",
                    retryCount, nbytes);
            Log(errs);
            continue;
            }
        
        /*
         ** Move past protocol and fill.
         */
        
        regResponse.index += 2;
        
        /*
         ** Get the result value from the response.
         */
        
        switch (result = GetWord(&regResponse))
            {
        case REGISTRATION_OK:
            
            if (nbytes >= REGISTRATION_RESPONSE_SIZE)
                {
                sprintf(errs, 
                        "Registered: %d reqs queued ahead.",
                        GetWord(&regResponse));
                Log(errs);
                }
            CancelTimer();
            close(regSocket);
            return;
            
        case REGISTRATION_QUEFULL:
            
            Log("Server Queue full, still trying.");
            break;
            
        default:
            
            sprintf(errs, "Server Error, result %d.", result);
            Log(errs);
            exit( EXIT_BAD );
            }
        }
    
    Log("Unable to register with server after %d tries.", retryCount);
    exit( EXIT_BAD );
    }

/*
 ** StartTimer()
 **
 ** Sets the timer to fire in the specified number of seconds.
 **
 ** Arguments:
 **
 **     seconds         The number of seconds for the timer to run.
 */

void
    StartTimer(seconds)
unsigned int    seconds;
{
    signal(SIGALRM, AlarmHandler);
    alarm(seconds);
    }

/*
 ** AlarmHandler()
 **
 ** This routine is defined so that SIGALRM signals may be caught.
 */

void
    AlarmHandler()
{
    if (config.debugLevel > 1)
        {
        Log("Handling SIGALRM.");
        }
    }

/*
 ** PipeHandler()
 **
 ** This routine is defined so that SIGPIPE signals may be caught.
 */

void
    PipeHandler()
{
    if (config.debugLevel > 1)
        {
        Log("Handling SIGPIPE.");
        }
    signal(SIGPIPE, PipeHandler);
    }

/*
 ** InterruptHandler()
 **
 ** This routine is defined so that SIGINT signals may be caught.
 */

void
    InterruptHandler()
{
    struct linger lingerer;
    int count;
    
    if (config.debugLevel > 1)
        {
        Log("Handling SIGINT or SIGTERM."); /* 020895 EKC/SAS SC45656 */
        }
    /*
     ** This may be abrupt, but we can't wait for things on the commServerFd
     ** it may cause us to hang up for a litlle bit.  We'll fix his wagon...
     */

    lingerer.l_onoff = 0;
    lingerer.l_linger = 0;
    if (setsockopt( config.commServerFd, SOL_SOCKET, SO_LINGER,
		   (char *) &lingerer, sizeof(lingerer)) == 0)
	{
	/*
	 ** There may not be a socket here.  This would happen only if
	 ** the connection was not initiated yet.  We don't want to print
	 ** any misleading error messages, so blow it off.
	 */
	shutdown( config.commServerFd, 2 );
	close (config.commServerFd);
	}
    
    /* 020895 EKC/SAS SC45656 If we have created a link with the filename, get rid of it. */

    if ( config.deviceFileName != NULL )
        unlink( config.deviceFileName );

    Log("Execution interrupted, program exiting");
    exit (0);
    }

/*
 ** CancelTimer()
 **
 ** Cancels the outstanding timer.
 */

void
    CancelTimer()
{
    alarm(0);
    }


/*
 ** GetWord()
 **
 ** Retrieves the next 16-bit word from packet, making sure that the
 ** end of the packet has not yet been reached, and updating the
 ** packet->index value to point to the next 16-bit word.
 **
 ** Arguments:
 **
 **     packet          The packet to read from.
 **
 ** Return value:
 **
 **     The next 16-bit word to be read from the packet.
 **
 ** Exits:
 **
 **     If an attempt is made to read past the end of the packet.
 */

unsigned short
    GetWord(packet)
Packet  *packet;
{
    unsigned short ret;
    
    if (packet->index > (packet->size - sizeof(unsigned short)))
        {
        Log("Tried to read past end of packet.");
        exit( EXIT_BAD );
        }
    
    ret = ntohs(*((unsigned short *)&packet->buffer[packet->index]));
    packet->index += sizeof(unsigned short);
    return(ret);
    }

/*
 ** PutChar()
 **
 ** Writes a character into a packet, making sure that the character
 ** is not written past the end of the packet, and updating the
 ** packet->index value to point to the next character and the
 ** packet->size value.
 **
 ** Arguments:
 **
 **     packet          The packet to write into.
 **
 **     c               The character to write.
 **
 ** Exits:
 **
 **     If an attempt is made to write past the end of the packet.
 */

void
    PutChar(packet, c)
Packet          *packet;
unsigned char    c;
{
    if (packet->index > PACKET_MAX-1)
        {
        Log("Tried to write past end of packet.");
        exit( EXIT_BAD );
        }
    
    packet->buffer[packet->index++] = c;
    packet->size++;
    }

/*
 ** PutWord()
 **
 ** Writes a 16-bit word into a packet, in network order, making
 ** sure that the word is not written past the end of the packet,
 ** and updating the packet->index value to point to the next 16-bit
 ** word and the packet->size value.
 **
 ** Arguments:
 **
 **     packet          The packet to write into.
 **
 **     w               The 16-bit word to write.
 **
 ** Exits:
 **
 **     If an attempt is made to write past the end of the packet.
 */

void
    PutWord(packet, w)
Packet          *packet;
unsigned short   w;
{
    unsigned short temp;
    
    if (packet->index > PACKET_MAX-1)
        {
        Log("Tried to write past end of packet.");
        exit( EXIT_BAD );
        }
    
    temp = htons(w);
    packet->buffer[packet->index++] = (temp & 0xff00) >> 8;
    packet->size++;
    packet->buffer[packet->index++] = temp & 0x00ff;
    packet->size++;
    }

/*
 ** ReadBuf()
 **
 ** Reads the specified file descriptor into contents of a buf,
 ** updating the buf pointers as data is read.
 **
 ** Arguments:
 **
 **     fd                      The file descriptor to read.
 **
 **     buf                     Ther buf to read into.
 **
 ** Return value:
 **
 **     0 on success, errno on read() failure, EIO when zero bytes read.
 */

int
    ReadBuf(fd, buf)
int      fd;
Buf     *buf;
{
    int count;
    int read_errno; /* 093094 GJG/SAS temp to store errno */

    if (config.debugLevel > 1)
        {
        Log("");
        sprintf(errs, "RB:  in: start = %d, end = %d",
                buf->start - buf->buffer,
                buf->end - buf->buffer);
        Log(errs);
        }

    count = read(fd, buf->end, BUFFER_ROOM( buf ));

    read_errno = errno; /* 093094 GJG/SAS Save errno so it doesn't get wiped out by the functions below */

    if (config.debugLevel > 1 )
        {
        sprintf(errs, "RB:      read %d bytes", count);
        Log(errs);
        }
    
    errno = read_errno; /* 093094 GJG/SAS Restore errno after the debug functions */
    
    if (count < 0)
        {

        if (config.debugLevel > 1 )
            {
            Log("");
            }
        
        if (errno != EIO)
            {
            LogSysError("ReadBuf: count < 0");
            }
        return(errno);
        }
    
    if (count == 0)
        {

        if (config.debugLevel > 1 )
            {
            Log("ReadBuf: count == 0"); /* 120694 SC43367 SAS Use Log not LogSysError here */
            }
        
        return(EIO);
        }
    
    *(buf->end + count) = 0;
    buf->end += count;

    if (config.debugLevel > 1 )
        {
        sprintf(errs, "RB: out: start = %d, end = %d",
                buf->start - buf->buffer,
                buf->end - buf->buffer);
        Log(errs);
        Log("");
        }

    return(0);
    }

/*
 ** WriteBuf()
 **
 ** Write the contents of a buf on the specified file descriptor,
 ** updating the buf pointers as data is written.
 **
 ** Arguments:
 **
 **     fd                      The file descriptor to write.
 **
 **     buf                     Ther buf to write from.
 **
 ** Return value:
 **
 **     0 on success, errno on write() failure.
 */

int
    WriteBuf(fd, buf)
int      fd;
Buf     *buf;
{
    int count;

    if (config.debugLevel > 1)
        {
        Log("");
        sprintf(errs, "WB:  in: start = %d, end = %d",
                buf->start - buf->buffer,
                buf->end - buf->buffer);
        Log(errs);
        }
    
	switch (*((unsigned char*)buf->start))
	{
		case '\176':
	    {
			if (tilde == 1)
			{
				count = write(fd, buf->start, buf->end - buf->start);
				tilde = 0;
			}
			else
			{
				tilde++;
				count = 1;
			}
			break;
		}
		case '\043':
	    {
			if (tilde == 1)
			{
				write(fd, "\377\363", 2);
				count = 1;
			}
			else
			{
				count = write(fd, buf->start, buf->end - buf->start);
			}
			tilde = 0;
			break;
		}
		default:
	    {
			count = write(fd, buf->start, buf->end - buf->start);
			tilde = 0;
			break;
		}
	}

    if (config.debugLevel > 1)
        {
        sprintf(errs, "WB:      wrote %d bytes", count);
        Log(errs);
        }
    
    if (count < 0)
        {
        
        if (config.debugLevel > 1)
            {
            Log("");
            }
        
        
        LogSysError("WriteBuf: count < 0");
        return(errno);
        }
    
    buf->start += count;
    
    if (buf->start == buf->end)
        {
        buf->start = buf->end = buf->buffer;
        }
    
    if (config.debugLevel > 1)
        {
        sprintf(errs, "WB: out: start = %d, end = %d",
                buf->start - buf->buffer,
                buf->end - buf->buffer);
        Log(errs);
        Log("");
        }
    
    return(0);
    }

/*
 ** TelnetOption()
 **
 ** Writes the specified telnet option string onto the serverSocket.
 **
 */

void TelnetOption( fd, commandChar, optionChar )
              int    fd;
     unsigned char   commandChar;
     unsigned char   optionChar;
{
    unsigned char        optBuf[4];
    unsigned char       *optPtr;
    int          optCnt;
    
    optPtr = optBuf;
    
    /*
     ** If the last character we wrote on the serverSocket was
     ** a TELNET_IAC_CHAR, don't write another one, just clear
     ** the flag so that we will double the one in the buf next
     ** time.  Otherwise, we need to write a TELNET_IAC_CHAR.
     */
    

    *optPtr++ = TELNET_IAC_CHAR;
    
    *optPtr++ = commandChar;
    *optPtr++ = optionChar;
    
    optCnt = optPtr - optBuf;
    
    sprintf(errs, "TelnetOption: writing");
    
    for (optPtr = optBuf; optPtr < (optBuf + optCnt); optPtr++)
        {
        sprintf(errs, "%s 0x%x", errs, *optPtr);
        }
    
    Log(errs);
    
    optPtr = optBuf;
    
    while (optCnt)
        {
        int     bytesWritten;
        
        bytesWritten = write(fd, optPtr, optCnt);
        
        if (bytesWritten == 0)
            {
            Log("TelnetOption: wrote 0 bytes");
            }
        else if (bytesWritten < 0)
            {
            LogSysError("TelnetOption: count < 0");
            exit( EXIT_BAD );
            }
        
        optCnt -= bytesWritten;
        optPtr += bytesWritten;
        }
    }
/*
 ** EorReflection()
 **
 ** Sends TELNET EOR to fd and waits for proper response.
 **
 ** Arguments:
 **
 **     fd                      A file descriptor -> Comm Server
 **
 ** Exits:
 **    
 **     If the Comm Server fails to respond properly.
 */
void EorReflection( fd )
     int   fd;
{
    int             bytesRcvd=0;
    unsigned char   ch = '\0', eorBuffer[2];

    sprintf( eorBuffer, "%c%c", 0xFF, 0xEF );
    Log( "Waiting for EOR reflection.");
    write( fd, eorBuffer, 2 );
    
    while( ch != 0xEF )
        {
        bytesRcvd = read( fd, (char *)&ch, 1 );
        
        if (bytesRcvd == 0)
            {
            LogSysError( "Read did not block." );
            exit( EXIT_BAD );
            }
        else if (bytesRcvd != 1)
            {
            Log( "Read returned too much data." );
            exit( EXIT_BAD );
            }
        }
    }
    
/*
 ** ConfigureUserIn()
 **
 ** Create a pipe, allocate a PTY or do nothing depending on what the
 ** user specified on the command line.  If a new file descriptor was
 ** created, then it will be "dup2'ed" into 0 (stdin).  When this
 ** function exits, stdin will be the host input source.
 **
 ** 
 */
void ConfigureUserIn()
{
    int     fd = 0, (*Link)();
    int     ON = 1, OFF = 0;
    
    switch( userSource )
        {
    case STANDARD_IN:
        config.userInFd = 0;
        break;
        
    case PTY:
        /*
         ** Input is set to come from a pseudo terminal (pty).  Allocate
         ** a pty and create a link to the slave side.  The name of the
         ** link is specified by the user.  When data is sent to this
         ** name, it will appear to us from the master side of the pty.
         */
        {
        if ((fd = AllocatePTY()) < 0)
            {
            Log( "Unable to allocate pty\n" );
            exit( EXIT_BAD );
            }

        /*
         ** Create a link to the pty.  If this machine doesn't support
         ** symlink(), then config.ptySymLink will be 0 and link() will
         ** be used.  We can be assured that config.ptySymLink is zero 
         ** because compile time switches cause the code associated with
         ** the -s option to be removed.  The user won't know about the
         ** option (and it's illegal if they try it) and won't be able
         ** to override the default value of config.ptySymLink flag which
         ** is FALSE.
         */
        
        sprintf( errs, "Removing device %s if it exists.", config.deviceFileName);
        Log( errs );
        unlink( config.deviceFileName );
        
        sprintf( errs, "Creating %s from %s to %s.",
                config.ptySymLink ? "symlink" : "link", config.ptySlaveName, 
                config.deviceFileName );
        Log( errs );
        
        Link = config.ptySymLink ? symlink : link;
        
        if ((*Link)( config.ptySlaveName, config.deviceFileName ))
            {
            sprintf( errs, "Unable to create %s for user device",
                    config.ptySymLink ? "symlink" : "link" );
            LogSysError( errs );
            exit( EXIT_BAD );
            }

#ifdef HPUX
	/* On HP-UX, for the master side to see a close on the slave side, this ioctl
	** must be done. See the HP-UX man page for pty for more info, and read it very
	** carefully!  Note that the master will only see the *final* close.
	*/
	if (ioctl( fd, TIOCTRAP, &ON) < 0)
	    {
	    LogSysError( "TIOCTRAP ioctl failed" );
	    exit( EXIT_BAD );
	    }
	if (config.debugLevel > 2)
	    {
	    Log( "Enabled trapping of open and close on user device" );
	    }
#endif

        break;
        }
        
    case PIPE:
        /*
         ** Input is set to come from a named pipe (FIFO).  Create the pipe
         ** (using the name specified) and open it for reading.  The read fd
         ** will be "dup2" into stdin.
         */
        {
        unlink( config.deviceFileName );
        if (mknod( config.deviceFileName, S_IFIFO | 0666, 0 ) < 0)
            {
            LogSysError( "Unable to create PIPE." );
            exit( EXIT_BAD );
            }
	if (chmod( config.deviceFileName, 0666) < 0)
            {
            LogSysError( "Unable to set PIPE access." );
            exit( EXIT_BAD );
            }
	/* PJB 5/16/97 SC104015/CR01103 - There is no need to delay connecting to the port. */
        if ((fd = open( config.deviceFileName, O_NDELAY  )) < 0)
            {
            LogSysError( "Unable to open PIPE." );
            exit( EXIT_BAD );
            }
        break;
        }
        
    default:
        Log( "Invalid host input source.\n" );
        exit( EXIT_BAD );
        break;
        }

    config.userInFd = fd;
    }

/*
 ** ConfigureUserOut()
 **
 ** Make stdout point to whatever the user desires.  Could be a pty, pipe or
 ** /dev/null if "write only mode" was asked for.
 */
void ConfigureUserOut(  )
{
    int     fd = 0;
    
    switch( userDestination )
        {
    case STANDARD_OUT:
        fd = 1;
        break;
        
    case PTY:
        /*
         ** User wants data from the Comm Server to be written to the PTY.  I
         ** hope their application is reading this data, or thinks could back
         ** up and might cause things to hang!.
         **
         ** ConfigureUserIn() left the fd number in the config structure for us.
         */
        {
        fd = config.userInFd;
        break;
        }
        
    case PIPE:
        /*
         ** Users wants data from the Comm Server to be written to a PIPE.
         ** ConfigureUserIn() created the pipe for us so all we need to do is
         ** open it for writing.
         */
        {
        if ((fd = open( config.deviceFileName, O_WRONLY  )) < 0)
            {
            LogSysError( "Unable to open PIPE." );
            exit( EXIT_BAD );
            }  
        
        break;
        }
        
        
    case DEV_NULL:
        {
        /*
         ** User doesn't want to see any data coming back from the Comm
         ** server.  Let's open /dev/null and and dup2 stdout to it!
         */
        
        if ((fd = open( "/dev/null", O_WRONLY )) < 0)
            {
            LogSysError( "Unable to open /dev/null\n");
            exit( EXIT_BAD );
            }
        
        break;
        
        }
        
    default:
        Log( "Invalid host input source.\n" );
        exit( EXIT_BAD );
        break;
        }

    config.userOutFd = fd;
    }    /*  end of ConfigureUserOut()  */


#if !defined(HAS_BCOPY)
/* bcopy()
 **
 ** Byte copy from s1 to s2 for len.
 */

void bcopy(s1, s2, len)
     char *s1, *s2;
     int  len;
{
    int i;

    for (i=0; i<len; i++)
        {
        *s2++ = *s1++;
        }
    }
#endif

#if !defined(HAS_BZERO)
/* bzero()
 **
 ** Zero out len bytes in s1.
 */

void bzero(s1, len)
     char *s1;
     int  len;
{
    int i;

    for (i=0; i<len; i++)
        {
        *s1++ = (char) 0;
        }
    }
#endif

/*****************************************************************************
 *****************************************************************************
 *****************     F i l t e r    F u n c t i o n s    *******************
 *****************************************************************************
 *****************************************************************************/
/*
 ** These functions process the data flowing between the user and the Comm
 ** Server port.  Each function accepts two arguments, a "from" buffer and a
 ** "to" buffer.  Data is read from the "from" buffer, processed and written
 ** to the "to" buffer.  Filter selection is handled by the function
 ** GetOptions().  
 */

/* 
** FilterNormToTelnet()
 **
 ** This filter converts "normal" data to "TELNET" data.  Data from the raw
 ** buffer is moved into the converted buffer.  Any TELNET_IAC_CHARs found
 ** are doubled.
 **
 ** Arguments: 
 **
 **     raw                      A buffer to read data into.
 **
 **     converted                A buffer to write data from.
 **
 **
 */

void FilterNormToTelnet( raw, converted )
     Buf *raw, *converted;

{
    
    if (!BUFFER_EMPTY( raw ))
	{
	/*
	 ** Double any TELNET_IAC_CHARs found.
	 */
	while ( (BUFFER_ROOM( converted ) > 1) &&
	       (!BUFFER_EMPTY( raw )) )

	/* 10/11/95 - SC65930 - JJ Added code to send a break sequence to terminal when in
                            limited Telnet Mode. */

      {
	    if ((unsigned char)*raw->start == TELNET_IAC_CHAR)
		{
		*converted->end++ = TELNET_IAC_CHAR;
        *converted->end++ = *raw->start++;
		}
	    else if ((config.breakMode) && ((unsigned char)*raw->start == config.breakChar))
        {
        *converted->end++ = TELNET_IAC_CHAR; 
		*converted->end++ = TELNET_BREAK_CHAR;
		raw->start++;
	    }
		
		else 
        {
         *converted->end++ = *raw->start++;	
	    }
	  }
        if (BUFFER_EMPTY( raw ))
	    {
	    BUFFER_RESET( raw );
	    }
	}
    }    /*  end of FilterNormToTelnet()  */

/*
 ** FilterTelnetToNorm()
 **
 ** This filter converts "TELNET" data to "normal" data.  Data from the raw
 ** buffer is moved into the converted buffer.  TELNET_IAC_CHARs are undoubled
 ** any options negotiation sequences are negatively acknowledged.
 **
 ** Arguments: 
 **
 **     raw                      A buffer to read data into.
 **
 **     converted                A buffer to write data from.
 **
 */

void FilterTelnetToNorm( raw, converted )
     Buf *raw, *converted;
     
{
    int              telnetReadState = TELNET_READ_NORMAL;
    
    if (!BUFFER_EMPTY( raw ))
	{
	/*
	 ** Undouble and TELNET_IAC_CHARs and negatively
	 ** acknowledge any option negotiations.
	 */
	while ( (BUFFER_ROOM( converted )) &&
	       (!BUFFER_EMPTY( raw )) )
	    {
	    switch (telnetReadState)
		{
	    case TELNET_READ_NORMAL:
		{
		if ((unsigned char)*raw->start == TELNET_IAC_CHAR)
		    {
		    telnetReadState = TELNET_READ_IAC;
		    raw->start++;
		    }
		else
		    {
		    *converted->end++ = *raw->start++;                      
		    }
		break;
		}
		
	    case TELNET_READ_IAC:
		{
		if ((unsigned char)*raw->start == TELNET_IAC_CHAR)
		    {
		    telnetReadState = TELNET_READ_NORMAL;
		    *converted->end++ = *raw->start++;
		    }
		else if ((unsigned char)*raw->start == TELNET_DO_CHAR)
		    {
		    telnetReadState = TELNET_READ_DO;
		    raw->start++;
		    }
		else if ((unsigned char)*raw->start == TELNET_WILL_CHAR)
		    {
		    telnetReadState = TELNET_READ_WILL;
		    raw->start++;
		    }
		else
		    {
		    if (config.debugLevel > 1)
			{
			sprintf( errs, "Unknown TELNET command '0x%x'",
				*raw->start);
			Log( errs );
			}
		    
		    raw->start++;
		    telnetReadState = TELNET_READ_NORMAL;
		    }
		break;
		}
		
	    case TELNET_READ_DO:
		{
		TelnetOption( config.commServerFd,  TELNET_WONT_CHAR, 
			     *raw->start++ );
		telnetReadState = TELNET_READ_NORMAL;
		break;
		}
		
	    case TELNET_READ_WILL:
		{
		TelnetOption( config.commServerFd, TELNET_DONT_CHAR, 
			     *raw->start++ );
		telnetReadState = TELNET_READ_NORMAL;
		break;
		}
                }
	    }
	
	
	if (BUFFER_EMPTY( raw ))
	    {
	    BUFFER_RESET( raw );
	    }
	}
    }    /*  end of FilterTelnetToNorm()  */



/*
 ** FilterCopy()
 **
 ** This filter simply copies data from the raw to the converted buffer.
 **
 ** Arguments: 
 **
 **     raw                      A buffer to read data into.
 **
 **     converted                A buffer to write data from.
 **
 **
 */

void FilterCopy( raw, converted )
     Buf *raw, *converted;
{
    
    if (!BUFFER_EMPTY( raw ))
	{
	while ( (BUFFER_ROOM( converted )) &&
	       (!BUFFER_EMPTY( raw )) )
                {
                *converted->end++ = *raw->start++;
                }
            
	if (BUFFER_EMPTY( raw ))
	    {
	    BUFFER_RESET( raw );
	    }
        }
    }    /*  end of FilterCopy()  */

/*
 ** FilterLfCr()
 **
 ** This filter converts LFs to LFCRs.  
 **
 ** Arguments: 
 **
 **     raw                      A buffer to read data into.
 **
 ** converted                    A buffer to write data from.
 **
 */

void FilterLfCr( raw, converted )
     Buf *raw, *converted;
{
    if (!BUFFER_EMPTY( raw ))
	{
	/*
	 ** Translate LF to LFCR as data is moved between the buffers.
	 */
	while ( (BUFFER_ROOM( converted ) > 1) &&
	       (!BUFFER_EMPTY( raw )) )
	    {
	    if (*raw->start == '\n')
		{
		*converted->end++ = '\r';
		}
	    *converted->end++ = *raw->start++;
	    }
	
	if (BUFFER_EMPTY( raw ))
	    {
	    BUFFER_RESET( raw );
	    }
	}
    }    /*  end of FilterLfCr()  */

